<?php

/**
 * webtrees: online genealogy
 * Copyright (C) 2025 webtrees development team
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <https://www.gnu.org/licenses/>.
 */

declare(strict_types=1);

namespace Fisharebest\Webtrees\Module;

use Fisharebest\Webtrees\FlashMessages;
use Fisharebest\Webtrees\I18N;
use Fisharebest\Webtrees\Validator;
use Psr\Http\Message\ResponseInterface;
use Psr\Http\Message\ServerRequestInterface;

/**
 * Class CustomCssJsModule - add CSS and JS to every page
 */
class CustomCssJsModule extends AbstractModule implements ModuleConfigInterface, ModuleGlobalInterface
{
    use ModuleConfigTrait;
    use ModuleGlobalTrait;

    /**
     * A sentence describing what this module does.
     *
     * @return string
     */
    public function description(): string
    {
        /* I18N: Description of the “CSS and JS” module. */
        return I18N::translate('Add styling and scripts to every page.');
    }

    public function isEnabledByDefault(): bool
    {
        return false;
    }

    /**
     * Show a form to edit the user CSS and JS.
     *
     * @param ServerRequestInterface $request
     *
     * @return ResponseInterface
     */
    public function getAdminAction(ServerRequestInterface $request): ResponseInterface
    {
        $this->layout = 'layouts/administration';

        return $this->viewResponse('modules/custom-css-js/edit', [
            'title' => $this->title(),
            'head'  => $this->getPreference('head'),
            'body'  => $this->getPreference('body'),
        ]);
    }

    public function title(): string
    {
        /* I18N: Name of a module. */
        return I18N::translate('CSS and JS');
    }

    /**
     * Save the user CSS and JS.
     *
     * @param ServerRequestInterface $request
     *
     * @return ResponseInterface
     */
    public function postAdminAction(ServerRequestInterface $request): ResponseInterface
    {
        $body = Validator::parsedBody($request)->string('body');
        $head = Validator::parsedBody($request)->string('head');

        $this->setPreference('body', $body);
        $this->setPreference('head', $head);

        $message = I18N::translate('The preferences for the module “%s” have been updated.', $this->title());
        FlashMessages::addMessage($message, 'success');

        return redirect($this->getConfigLink());
    }

    /**
     * Raw content, to be added at the end of the <body> element.
     * Typically, this will be <script> elements.
     *
     * @return string
     */
    public function bodyContent(): string
    {
        return $this->getPreference('body');
    }

    /**
     * Raw content, to be added at the end of the <head> element.
     * Typically, this will be <link> and <meta> elements.
     *
     * @return string
     */
    public function headContent(): string
    {
        return $this->getPreference('head');
    }
}
