<?php

/*
 * This file is part of the TYPO3 CMS project.
 *
 * It is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License, either version 2
 * of the License, or any later version.
 *
 * For the full copyright and license information, please read the
 * LICENSE.txt file that was distributed with this source code.
 *
 * The TYPO3 project - inspiring people to share!
 */

namespace TYPO3\CMS\Backend\Tree\View;

use Psr\Http\Message\ServerRequestInterface;
use TYPO3\CMS\Backend\Routing\UriBuilder;
use TYPO3\CMS\Core\Utility\GeneralUtility;

/**
 * Position map class for moving pages.
 *
 * @internal This class is a TYPO3 Backend implementation and is not considered part of the Public TYPO3 API.
 */
class PageMovingPagePositionMap extends PagePositionMap
{
    /**
     * Creates the link target for the insert-icons.
     *
     * @param int $pid The pid.
     * @param int $newPagePID New page id.
     * @return string Onclick attribute content
     */
    public function getActionLink($pid, $newPagePID): string
    {
        $uriBuilder = GeneralUtility::makeInstance(UriBuilder::class);
        return (string)$uriBuilder->buildUriFromRoute('tce_db', [
            'cmd[pages][' . $this->moveUid . '][' . $this->moveOrCopy . ']' => $pid,
            'redirect' => $this->R_URI,
        ]);
    }

    /**
     * Wrapping page title.
     *
     * @param string $str Page title.
     * @param array $rec Page record (?)
     * @return string Wrapped title.
     */
    public function linkPageTitle($str, $rec, ServerRequestInterface $request)
    {
        $uriBuilder = GeneralUtility::makeInstance(UriBuilder::class);
        $queryParams = $request->getQueryParams();
        $url = $uriBuilder->buildUriFromRoute(
            'move_element',
            [
                'table' => $queryParams['table'] ?? '',
                'uid' => (int)($rec['uid'] ?? 0),
                'moveUid' => $this->moveUid,
                'makeCopy' => $queryParams['makeCopy'] ?? 0,
                'returnUrl' => $queryParams['returnUrl'] ?? '',
            ]
        );
        return '<a href="' . htmlspecialchars($url) . '">' . $str . '</a>';
    }

    /**
     * Wrap $t_code in bold IF the $dat uid matches $id
     *
     * @param string $t_code Title string
     * @param array $dat Information array with record array inside.
     * @param int $id The current id.
     * @return string The title string.
     */
    public function boldTitle($t_code, $dat, $id)
    {
        return parent::boldTitle($t_code, $dat, (int)$this->moveUid);
    }

    /**
     * Get label, htmlspecialchars()'ed
     *
     * @return string The localized label for "move page here"
     */
    protected function insertlabel()
    {
        return htmlspecialchars($this->getLanguageService()->sL('LLL:EXT:core/Resources/Private/Language/locallang_misc.xlf:movePageToHere'));
    }
}
