<?php

// (c) Copyright by authors of the Tiki Wiki CMS Groupware Project
//
// All Rights Reserved. See copyright.txt for details and a complete list of authors.
// Licensed under the GNU LESSER GENERAL PUBLIC LICENSE. See license.txt for details.
namespace Tracker\Tabular;

use Exception;
use Feedback;

class Manager
{
    private $table;

    public function __construct(\TikiDb $db)
    {
        $this->table = $db->table('tiki_tabular_formats');
    }

    public function getList($conditions = [])
    {
        return $this->table->fetchAll(['tabularId', 'name', 'trackerId'], $conditions, -1, -1, 'name_asc');
    }

    public function getInfo($tabularId)
    {
        $info = $this->table->fetchFullRow(['tabularId' => $tabularId]);

        $info['format_descriptor'] = ! empty($info['format_descriptor']) ? json_decode($info['format_descriptor'], true) : [];
        $info['filter_descriptor'] = ! empty($info['filter_descriptor']) ? json_decode($info['filter_descriptor'], true) : [];
        $info['config'] = ! empty($info['config']) ? json_decode($info['config'], true) : [];
        $info['odbc_config'] = ! empty($info['odbc_config']) ? json_decode($info['odbc_config'], true) : [];
        $info['api_config'] = ! empty($info['api_config']) ? json_decode($info['api_config'], true) : [];
        return $info;
    }

    public function create($name, $trackerId, $odbc_config = [])
    {
        $this->validateOdbcConfig($odbc_config, []);
        return $this->table->insert([
            'name' => $name,
            'trackerId' => $trackerId,
            'format_descriptor' => '[]',
            'filter_descriptor' => '[]',
            'config' => json_encode([
                'simple_headers' => 0,
                'import_update' => 1,
                'ignore_blanks' => 0,
                'import_transaction' => 0,
                'bulk_import' => 0,
                'skip_unmodified' => 0,
                'skip_validation' => 0,
                'encoding' => '',
                'format' => '',
                'mapping' => '',
            ]),
            'odbc_config' => json_encode($odbc_config),
        ]);
    }

    public function update($tabularId, $name, array $fields, array $filters, array $config, array $odbc_config = [], array $api_config = [])
    {
        $info = $this->table->fetchFullRow(['tabularId' => $tabularId]);
        $info['odbc_config'] = ! empty($info['odbc_config']) ? json_decode($info['odbc_config'], true) : [];
        $this->validateOdbcConfig($odbc_config, $info['odbc_config']);
        return $this->table->update([
            'name' => $name,
            'format_descriptor' => json_encode($fields),
            'filter_descriptor' => json_encode($filters),
            'config' => json_encode([
                'simple_headers' => (int)! empty($config['simple_headers']),
                'import_update' => (int)! empty($config['import_update']),
                'ignore_blanks' => (int)! empty($config['ignore_blanks']),
                'import_transaction' => (int)! empty($config['import_transaction']),
                'bulk_import' => (int)! empty($config['bulk_import']),
                'skip_unmodified' => (int)! empty($config['skip_unmodified']),
                'skip_validation' => (int)! empty($config['skip_validation']),
                'encoding' => $config['encoding'] ?? '',
                'format' => $config['format'] ?? '',
                'mapping' => $config['mapping'] ?? '',
            ]),
            'odbc_config' => json_encode($odbc_config),
            'api_config' => json_encode($api_config)
        ], ['tabularId' => $tabularId]);
    }

    public function remove($tabularId)
    {
        return $this->table->delete(['tabularId' => $tabularId]);
    }

    public function syncItemSaved($args)
    {
        if (isset($args['skip_sync']) && $args['skip_sync']) {
            return;
        }

        $trklib = \TikiLib::lib('trk');

        $definition = \Tracker_Definition::get($args['trackerId']);
        $tabulars = [];
        try {
            $tabulars = $definition->getSynchronizedTabulars();
        } catch (Exception $e) {
            Feedback::error($e->getMessage());
            return;
        }

        foreach ($tabulars as $tabular) {
            $schema = $this->getSchema($definition, $tabular);

            try {
                if ($tabular['odbc_config']) {
                    $source = new \Tracker\Tabular\Source\TrackerItemSource($schema, $args['object']);
                    $writer = new Writer\ODBCWriter($tabular['odbc_config']);
                    $remote = $writer->sync($source, $args['object'], $args['old_values_by_permname'], $args['values_by_permname'], $is_new);
                    foreach ($remote as $field => $value) {
                        if (isset($args['values_by_permname'][$field])) {
                            $differs = $value !== $args['values_by_permname'][$field];
                        } elseif ($is_new) {
                            $differs = true;
                        } else {
                            $differs = false;
                        }
                        if ($differs) {
                            $field = $definition->getFieldFromPermName($field);
                            $trklib->modify_field($args['object'], $field['fieldId'], $value);
                        }
                    }
                } elseif ($tabular['api_config']) {
                    global $jitRequest;
                    $remote_url = $jitRequest->tiki_skip_sync_url->raw();
                    if (TIKI_API && ! empty($tabular['api_config']['update_url']) && ! empty($remote_url) && stristr($tabular['api_config']['update_url'], $remote_url)) {
                        // skip syncing back changes coming from the target host via the API
                        continue;
                    }
                    $source = new \Tracker\Tabular\Source\TrackerItemSource($schema, $args['object']);
                    $writer = new \Tracker\Tabular\Writer\APIWriter($tabular['api_config'], $tabular['config']);
                    $result = $writer->write($source);
                    if (! empty($result['errors'])) {
                        throw new Exception($result['errors'][0]);
                    }
                }
            } catch (Exception $e) {
                Feedback::error(tr("Failed synchronizing local changes with remote data source. Please try making these changes again later or make the same changes remotely. Error: %0", $e->getMessage()));
            }
        }
    }

    public function syncItemDeleted($args)
    {
        if (isset($args['skip_sync']) && $args['skip_sync']) {
            return;
        }

        $definition = \Tracker_Definition::get($args['trackerId']);
        $tabulars = [];
        try {
            $tabulars = $definition->getSynchronizedTabulars();
        } catch (Exception $e) {
            Feedback::error($e->getMessage());
            return;
        }

        foreach ($tabulars as $tabular) {
            $schema = $this->getSchema($definition, $tabular);

            if ($tabular['odbc_config']) {
                if (empty($tabular['odbc_config']['sync_deletes'])) {
                    continue;
                }
                foreach ($schema->getColumns() as $column) {
                    if ($column->isPrimaryKey()) {
                        $field = $definition->getFieldFromPermName($column->getField());
                        $id = $args['values'][$field['fieldId']] ?: null;
                        if ($id) {
                            try {
                                $writer = new Writer\ODBCWriter($tabular['odbc_config']);
                                $writer->delete($column->getRemoteField(), $id);
                            } catch (Exception $e) {
                                Feedback::error(tr("Failed synchronizing local item delete with remote data source. Remote item might get reimported. Please try deleting again later or delete the item remotely. Error: %0", $e->getMessage()));
                            }
                            break;
                        }
                    }
                }
            } elseif ($tabular['api_config']) {
                global $jitRequest;
                $remote_url = $jitRequest->tiki_skip_sync_url->raw();
                if (TIKI_API && ! empty($tabular['api_config']['delete_url']) && ! empty($remote_url) && stristr($tabular['api_config']['delete_url'], $remote_url)) {
                    // skip syncing back changes coming from the target host via the API
                    continue;
                }
                $source = new \Tracker\Tabular\Source\TrackerItemSource($schema, null, $args['values']);
                $writer = new \Tracker\Tabular\Writer\APIWriter($tabular['api_config'], $tabular['config']);
                $result = $writer->write($source, 'delete');
                if (! empty($result['errors'])) {
                    Feedback::error($result['errors'][0]);
                    continue;
                }
            }
        }
    }

    public function syncCommentSaved($args)
    {
        if (isset($args['skip_sync']) && $args['skip_sync']) {
            return;
        }

        if (empty($args['parentobject'])) {
            return;
        }

        $trklib = \TikiLib::lib('trk');

        $definition = \Tracker_Definition::get($args['parentobject']);
        $tabulars = [];
        try {
            $tabulars = $definition->getSynchronizedTabulars();
        } catch (Exception $e) {
            Feedback::error($e->getMessage());
            return;
        }

        foreach ($tabulars as $tabular) {
            $schema = $this->getSchema($definition, $tabular);
            if ($tabular['api_config']) {
                $source = new \Tracker\Tabular\Source\TrackerItemSource($schema, $args['object']);
                $writer = new \Tracker\Tabular\Writer\APIWriter($tabular['api_config'], $tabular['config']);
                $writer->writeComment($args, $source);
            }
        }
    }

    public function syncCommentDeleted($args)
    {
        if (isset($args['skip_sync']) && $args['skip_sync']) {
            return;
        }

        if (empty($args['parentobject'])) {
            return;
        }

        $definition = \Tracker_Definition::get($args['parentobject']);
        $tabulars = [];
        try {
            $tabulars = $definition->getSynchronizedTabulars();
        } catch (Exception $e) {
            Feedback::error($e->getMessage());
            return;
        }

        foreach ($tabulars as $tabular) {
            $schema = $this->getSchema($definition, $tabular);
            if ($tabular['api_config']) {
                $source = new \Tracker\Tabular\Source\TrackerItemSource($schema, $args['object']);
                $writer = new \Tracker\Tabular\Writer\APIWriter($tabular['api_config'], $tabular['config']);
                $writer->writeComment($args, $source, 'delete');
            }
        }
    }

    public function getSchema($definition, $tabular)
    {
        $schema = new Schema($definition);
        $schema->loadFormatDescriptor($tabular['format_descriptor']);
        $schema->loadFilterDescriptor($tabular['filter_descriptor']);
        $schema->loadConfig($tabular['config']);

        return $schema;
    }

    public function validateRemoteUnchanged($trackerId, $itemId)
    {
        $definition = \Tracker_Definition::get($trackerId);
        if (! $definition) {
            Feedback::error(tr("Tracker not found: %0", $trackerId));
            return true;
        }

        $tabulars = [];
        try {
            $tabulars = $definition->getSynchronizedTabulars('odbc');
        } catch (Exception $e) {
            Feedback::error($e->getMessage());
            return true;
        }

        if (empty($tabulars)) {
            Feedback::error(tr("Tracker not configured for remote synchronization: %0", $trackerId));
            return true;
        }

        foreach ($tabulars as $tabular) {
            $schema = $this->getSchema($definition, $tabular);
            $item = \Tracker_Item::fromId($itemId);
            $item = $item->getData();
            $item = $item['fields'];

            try {
                $writer = new Writer\ODBCWriter($tabular['odbc_config']);
                $diff = $writer->compareRemote($schema, $itemId, $item);

                if ($diff) {
                    $error = tr("Remote item has changed since your last page load. Overwriting remote data is disabled. You can copy your changes to a safe place, reload the entry and make the changes again. Here's the difference:");
                    $error .= "\n" . tr("Field | Local | Remote");
                    foreach ($diff as $permName => $value) {
                        $field = $definition->getFieldFromPermName($permName);
                        \TikiLib::lib('trk')->modify_field($itemId, $field['fieldId'], $value);
                        $local = $item[$permName];
                        if (is_array($local)) {
                            $local = implode(',', $local);
                        }
                        $error .= "\n" . $field['name'] . ' | ' . $local . ' | ' . $value;
                    }
                    return $error;
                } else {
                    return true;
                }
            } catch (Exception $e) {
                return tr("Failed ensuring remote item is up to date with local data. Please check remote server connectivity and try again. Error: %0", $e->getMessage());
            }
        }
    }

    protected function validateOdbcConfig(&$odbc_config, $old_config): void
    {
        try {
            if (! empty($odbc_config['permanent_values'])) {
                $odbc_config['permanent_values'] = json_decode($odbc_config['permanent_values'], true, 512, JSON_THROW_ON_ERROR);
                if (! is_array($odbc_config['permanent_values'])) {
                    throw new Exception('invalid format');
                }
                foreach ($odbc_config['permanent_values'] as $key => $val) {
                    if (! is_string($key) || ! is_scalar($val)) {
                        throw new Exception('invalid format');
                    }
                }
            }
        } catch (Exception $e) {
            Feedback::error(tr("Failed parsing Remote Permanent Values field: %0. Changes were not saved.", $e->getMessage()));
            $odbc_config = $old_config;
            return;
        }
        try {
            if (! empty($odbc_config['value_mappings'])) {
                $odbc_config['value_mappings'] = @json_decode($odbc_config['value_mappings'], true, 512, JSON_THROW_ON_ERROR);
                if (! is_array($odbc_config['value_mappings'])) {
                    throw new Exception('invalid format');
                }
                foreach ($odbc_config['value_mappings'] as $field => $mapping) {
                    if (! is_string($field) || ! is_array($mapping)) {
                        throw new Exception('invalid format');
                    }
                    foreach ($mapping as $remote => $local) {
                        if (! is_scalar($remote) || (! is_scalar($local) && ! is_array($local))) {
                            throw new Exception('invalid format');
                        }
                    }
                }
            }
        } catch (Exception $e) {
            Feedback::error(tr("Failed parsing Field Value Mappings field: %0. Changes were not saved.", $e->getMessage()));
            $odbc_config = $old_config;
            return;
        }
        try {
            if (! empty($odbc_config['join_tables'])) {
                $odbc_config['join_tables'] = json_decode($odbc_config['join_tables'], true, 512, JSON_THROW_ON_ERROR);
                if (! is_array($odbc_config['join_tables'])) {
                    throw new Exception('invalid format');
                }
                foreach ($odbc_config['join_tables'] as $key => $val) {
                    if (! is_string($key)) {
                        throw new Exception('invalid format');
                    }
                    if (! is_array($val)) {
                        throw new Exception('invalid format');
                    }
                    foreach ($val as $remote => $local) {
                        if (! is_scalar($remote) || ! is_scalar($local)) {
                            throw new Exception('invalid format');
                        }
                    }
                }
            }
        } catch (Exception $e) {
            Feedback::error(tr("Failed parsing Join Tables field: %0. Changes were not saved.", $e->getMessage()));
            $odbc_config = $old_config;
            return;
        }
    }
}
