<?php

// (c) Copyright by authors of the Tiki Wiki CMS Groupware Project
//
// All Rights Reserved. See copyright.txt for details and a complete list of authors.
// Licensed under the GNU LESSER GENERAL PUBLIC LICENSE. See license.txt for details.
class Search_MySql_Table extends TikiDb_Table
{
    public const MAX_MYSQL_INDEXES_PER_TABLE = 64;

    private $definition = false;
    private $indexes = [];
    private $exists = null;

    private $schemaBuffer;
    private $dataBuffer;
    private $tfTranslator;

    public function __construct($db, $table)
    {
        parent::__construct($db, $table);

        $table = $this->escapeIdentifier($this->tableName);
        $this->schemaBuffer = new Search_MySql_QueryBuffer($db, 2000, "ALTER TABLE $table ");
        $this->dataBuffer = new Search_MySql_QueryBuffer($db, 100, '-- '); // Null Object, replaced later
        $this->tfTranslator = new Search_MySql_TrackerFieldTranslator();
    }

    public function __destruct()
    {
        try {
            $this->flush();
        } catch (Search_MySql_Exception $e) {
            # ignore this to cleanly destruct the object
        }
    }

    public function drop()
    {
        $table = $this->escapeIdentifier($this->tableName);
        $this->db->query("DROP TABLE IF EXISTS $table");
        $this->definition = false;
        $this->exists = false;

        $this->emptyBuffer();
    }

    public function exists()
    {
        if (is_null($this->exists)) {
            $tables = $this->db->listTables();
            $this->exists = in_array($this->tableName, $tables);
        }

        return $this->exists;
    }

    public function insert(array $values, $ignore = false)
    {
        $keySet = implode(', ', array_map([$this, 'escapeIdentifier'], array_map([$this->tfTranslator, 'shortenize'], array_keys($values))));

        $valueSet = '(' . implode(', ', array_map([$this->db, 'qstr'], $values)) . ')';

        $this->addToBuffer($keySet, $valueSet);

        return 0;
    }

    public function ensureHasField($fieldName, $type)
    {
        $this->loadDefinition();

        if (! isset($this->definition[$fieldName])) {
            $this->addField($fieldName, $type);
            $this->definition[$fieldName] = $type;
        }
    }

    public function hasIndex($fieldName, $type)
    {
        $this->loadDefinition();

        $indexName = $fieldName . '_' . $type;
        return isset($this->indexes[$indexName]);
    }

    /**
     * Make sure the indexing table contains a certain index. Index will only be added if it is not present on the table.
     * @param $fieldName
     * @param $type
     * @throws Search_MySql_QueryException
     */
    public function ensureHasIndex($fieldName, $type)
    {
        global $prefs;

        $this->loadDefinition();
        $fieldName = $this->tfTranslator->normalize($fieldName);

        if (! isset($this->definition[$fieldName])) {
            if (preg_match('/^tracker_field_/', $fieldName)) {
                $msg = tr('Field %0 does not exist in the current index. Please check field permanent name and if you have any items in that tracker.', TikiFilter::get('xss')->filter($fieldName));
                if ($prefs['unified_exclude_nonsearchable_fields'] === 'y') {
                    $msg .= ' ' . tr('You have disabled indexing non-searchable tracker fields. Check if this field is marked as searchable.');
                }
            } else {
                $msg = tr('Field %0 does not exist in the current index. If this is a tracker field, the proper syntax is tracker_field_%0.', TikiFilter::get('xss')->filter($fieldName), TikiFilter::get('xss')->filter($fieldName));
            }
            $e = new Search_MySql_QueryException($msg);
            if ($fieldName == 'tracker_id' || $prefs['search_error_missing_field'] !== 'y') {
                $e->suppress_feedback = true;
            }
            throw $e;
        }

        $indexName = $fieldName . '_' . $type;

        // Static MySQL limit on 64 indexes per table
        if (! isset($this->indexes[$indexName]) && count($this->indexes) < self::MAX_MYSQL_INDEXES_PER_TABLE) {
            if ($type == 'fulltext') {
                $this->addFullText($fieldName);
            } elseif ($type == 'index') {
                $this->addIndex($fieldName);
            }

            $this->indexes[$indexName] = true;
        }
    }

    public function getFieldType($fieldName)
    {
        $this->loadDefinition();
        if (isset($this->definition[$fieldName])) {
            return $this->definition[$fieldName];
        }
        return null;
    }

    /**
     * Fetch results from the index but leave the fields relevant for each document type
     * to reduce memory footprint for big indices.
     * @param array $fields
     * @param array $conditions
     * @param int $numrows
     * @param int $offset
     * @param string $orderClause
     * @return array of associative arrays
     */
    public function fetchAllIndex(array $fields = [], array $conditions = [], $numrows = -1, $offset = -1, $orderClause = null)
    {
        $available_fields = TikiLib::lib('unifiedsearch')->getAvailableFields();
        $resultset = $this->query($fields, $conditions, $numrows, $offset, $orderClause);
        $result = [];
        while ($row = $resultset->fetchRow()) {
            if ($row['object_type'] == 'trackeritem') {
                $fields = $available_fields['object_types']['trackeritem' . $row['tracker_id']] ?? [];
            } else {
                $fields = $available_fields['object_types'][$row['object_type']] ?? [];
            }
            if ($fields) {
                $real_row = [];
                foreach ($fields as $field) {
                    $field = $this->tfTranslator->shortenize($field);
                    foreach ($row as $key => $value) {
                        if (str_starts_with($key, $field)) {
                            $real_row[$key] = $row[$key];
                        }
                    }
                }
                $result[] = $real_row;
            } else {
                $result[] = $row;
            }
        }
        return $result;
    }

    private function loadDefinition()
    {
        if (! empty($this->definition)) {
            return;
        }

        if (! $this->exists()) {
            $this->createTable();
            $this->loadDefinition();
        }

        $table = $this->escapeIdentifier($this->tableName);
        $result = $this->db->fetchAll("DESC $table");
        $this->definition = [];
        foreach ($result as $row) {
            $this->definition[$this->tfTranslator->normalize($row['Field'])] = $row['Type'];
        }

        $result = $this->db->fetchAll("SHOW INDEXES FROM $table");
        $this->indexes = [];
        foreach ($result as $row) {
            $this->indexes[$this->tfTranslator->normalize($row['Key_name'])] = true;
        }
    }

    private function createTable()
    {
        $table = $this->escapeIdentifier($this->tableName);
        $this->db->query(
            "CREATE TABLE IF NOT EXISTS $table (
                `id` INT NOT NULL AUTO_INCREMENT,
                `object_type` VARCHAR(15) NOT NULL,
                `object_id` VARCHAR(235) NOT NULL,
                PRIMARY KEY(`id`),
                INDEX (`object_type`, `object_id`(160))
            ) ENGINE=MyISAM"
        );
        $this->exists = true;

        $this->emptyBuffer();
    }

    private function addField($fieldName, $type)
    {
        $table = $this->escapeIdentifier($this->tableName);
        $fieldName = $this->escapeIdentifier($this->tfTranslator->shortenize($fieldName));
        $this->schemaBuffer->push("ADD COLUMN $fieldName $type");
    }

    private function addIndex($fieldName)
    {
        $currentType = $this->definition[$fieldName];
        $alterType = null;

        $indexName = $fieldName . '_index';
        $table = $this->escapeIdentifier($this->tableName);
        $escapedIndex = $this->escapeIdentifier($this->tfTranslator->shortenize($indexName));
        $escapedField = $this->escapeIdentifier($this->tfTranslator->shortenize($fieldName));

        if ($currentType == 'TEXT' || $currentType == 'text') {
            $this->schemaBuffer->push("MODIFY COLUMN $escapedField VARCHAR(235)");
            $this->definition[$fieldName] = 'VARCHAR(235)';
        }

        $this->schemaBuffer->push("ADD INDEX $escapedIndex ($escapedField)");
    }

    private function addFullText($fieldName)
    {
        $indexName = $fieldName . '_fulltext';
        $table = $this->escapeIdentifier($this->tableName);
        $escapedIndex = $this->escapeIdentifier($this->tfTranslator->shortenize($indexName));
        $escapedField = $this->escapeIdentifier($this->tfTranslator->shortenize($fieldName));
        $this->schemaBuffer->push("ADD FULLTEXT INDEX $escapedIndex ($escapedField)");
    }

    private function emptyBuffer()
    {
        $this->schemaBuffer->clear();
        $this->dataBuffer->clear();
    }

    private function addToBuffer($keySet, $valueSet)
    {
        $this->schemaBuffer->flush();

        $this->dataBuffer->setPrefix("INSERT INTO {$this->escapeIdentifier($this->tableName)} ($keySet) VALUES ");
        $this->dataBuffer->push($valueSet);
    }

    public function flush()
    {
        $this->schemaBuffer->flush();
        $this->dataBuffer->flush();
    }
}
