# Movable Type (r) (C) Six Apart Ltd. All Rights Reserved.
# This code cannot be redistributed without permission from www.sixapart.com.
# For more information, consult your Movable Type license.
#
package MT::Image::ExifData;

use strict;
use warnings;

my %WritableTags;

sub writable_tags {
    my $group = shift;
    if (!%WritableTags) {
        while (<DATA>) {
            chomp;
            next if /^(#|\s)/;
            $WritableTags{$_} = 1;
        }
    }
    \%WritableTags;
}

#----------------------------------------------------------------------------

sub update_me {
    my ($class, %args) = @_;
    _require_module('Perl::Tidy') or return;

    my $file = __FILE__;
    my $body = do { open my $fh, '<', $file or die $!; local $/; <$fh> };
    $body =~ s/(__DATA__\n).*$/$1/s;

    require Image::ExifTool;
    require Image::ExifTool::Exif;

    $body .= "# Generated from Image::ExifTool $Image::ExifTool::VERSION\n";

    my $exif = Image::ExifTool->new;
    $exif->Options(Composite  => 0);
    $exif->Options(Duplicates => 0);

    my %ignore;
    for my $value (values %Image::ExifTool::Exif::Main) {
        if (ref $value eq 'HASH' && ($value->{Mandatory} || $value->{Protected})) {
            $ignore{ $value->{Name} } = 1;
        }
    }

    my %tags;
    for my $group (qw(EXIF XMP)) {
        for my $tag (Image::ExifTool::GetWritableTags($group)) {
            next if $ignore{$tag};
            $tags{$tag} = 1;
        }
    }

    for my $tag (sort keys %tags) {
        $body .= "$tag\n";
    }

    Perl::Tidy::perltidy(
        source      => \$body,
        destination => $file,
    );
}

sub _require_module {
    my $module = shift;
    eval "require $module; 1" or do {
        print STDERR "self-updating requires $module\n";
        return;
    };
}

1;

__DATA__
# Generated from Image::ExifTool 12.76
About
AboutCvTerm
AboutCvTermCvId
AboutCvTermId
AboutCvTermName
AboutCvTermRefinedAbout
AbsPeakAudioFilePath
AbsoluteAltitude
AcademicField
Acceleration
AdditionalModelInformation
AdultContentWarning
Advisory
AeProjectLink
AeProjectLinkCompositionID
AeProjectLinkFullPath
AeProjectLinkRenderOutputModuleIndex
AeProjectLinkRenderQueueItemID
AeProjectLinkRenderTimeStamp
AffineA
AffineB
AffineC
AffineD
AffineX
AffineY
AggregateIssueNumber
AggregationType
Agreement
Album
AlreadyApplied
AltTapeName
AltTextAccessibility
AltTimecode
AltTimecodeTimeFormat
AltTimecodeTimeValue
AltTimecodeValue
AlternateTitle
AlternateTitleA-lang
AlternateTitleA-platform
AlternateTitleText
AmbientTemperature
AngleInfoRoll
AngleInfoYaw
ApertureValue
AppInfo
AppInfoApplication
AppInfoItemURI
AppInfoVersion
AppVersion
ApplyAutomatically
Approved
ApprovedBy
ApproximateFocusDistance
Artist
ArtworkCircaDateCreated
ArtworkContentDescription
ArtworkContributionDescription
ArtworkCopyrightNotice
ArtworkCopyrightOwnerID
ArtworkCopyrightOwnerName
ArtworkCreator
ArtworkCreatorID
ArtworkDateCreated
ArtworkLicensorID
ArtworkLicensorName
ArtworkOrObject
ArtworkPhysicalDescription
ArtworkSource
ArtworkSourceInvURL
ArtworkSourceInventoryNo
ArtworkStylePeriod
ArtworkTitle
AssetID
AttributionName
AttributionURL
AudioBitrate
AudioBitrateMode
AudioBitsPerSample
AudioChannelCount
AudioChannelType
AudioCompressor
AudioData
AudioMimeType
AudioModDate
AudioSampleRate
AudioSampleType
Author
AuthorsPosition
AutoBrightness
AutoContrast
AutoExposure
AutoLateralCA
AutoShadows
AutoTone
AutoToneDigest
AutoToneDigestNoSat
AutoWhiteVersion
BackgroundAlpha
BaseRenditionIsHDR
BaseURL
BeatSpliceParams
BeatSpliceParamsRiseInDecibel
BeatSpliceParamsRiseInTimeDuration
BeatSpliceParamsRiseInTimeDurationScale
BeatSpliceParamsRiseInTimeDurationValue
BeatSpliceParamsUseFileBeatsMarker
BlackLevelBlue
BlackLevelGreen
BlackLevelRed
Blacks2012
BlogTitle
BlogURL
BlueBalance
BlueHue
BlueSaturation
BlurAtInfinity
BookEdition
Brightness
BrightnessValue
BurstID
BurstPrimary
ByteCount
CCVAvgLuminanceNits
CCVMaxLuminanceNits
CCVMinLuminanceNits
CCVPrimariesXY
CCVWhiteXY
CFAPattern
CFAPatternColumns
CFAPatternRows
CFAPatternValues
CRC32
CalibratedFocalLength
CalibratedOpticalCenterX
CalibratedOpticalCenterY
CallForImage
CamReverse
Camera
CameraAngle
CameraAppInfo
CameraAppInfoApplication
CameraAppInfoItemURI
CameraAppInfoVersion
CameraBurstID
CameraDepthMap
CameraDepthMapConfidenceURI
CameraDepthMapDepthURI
CameraDepthMapFar
CameraDepthMapFocalTable
CameraDepthMapFocalTableEntryCount
CameraDepthMapFormat
CameraDepthMapItemSemantic
CameraDepthMapMeasureType
CameraDepthMapNear
CameraDepthMapSoftware
CameraDepthMapUnits
CameraElevationAngle
CameraFilename
CameraFirmware
CameraImage
CameraImageItemSemantic
CameraImageItemURI
CameraImagingModel
CameraImagingModelDistortion
CameraImagingModelDistortionCount
CameraImagingModelFocalLengthX
CameraImagingModelFocalLengthY
CameraImagingModelImageHeight
CameraImagingModelImageWidth
CameraImagingModelPixelAspectRatio
CameraImagingModelPrincipalPointX
CameraImagingModelPrincipalPointY
CameraImagingModelSkew
CameraLabel
CameraLightEstimate
CameraLightEstimateColorCorrectionB
CameraLightEstimateColorCorrectionG
CameraLightEstimateColorCorrectionR
CameraLightEstimatePixelIntensity
CameraMakeModel
CameraModel
CameraModelID
CameraModelRestriction
CameraMove
CameraPointCloud
CameraPointCloudMetric
CameraPointCloudPointCloud
CameraPointCloudPoints
CameraPose
CameraPosePositionX
CameraPosePositionY
CameraPosePositionZ
CameraPoseRotationW
CameraPoseRotationX
CameraPoseRotationY
CameraPoseRotationZ
CameraPoseTimestamp
CameraProfile
CameraProfileDigest
CameraProfiles
CameraProfilesApertureValue
CameraProfilesAuthor
CameraProfilesAutoScale
CameraProfilesCameraPrettyName
CameraProfilesCameraRawProfile
CameraProfilesFocalLength
CameraProfilesFocusDistance
CameraProfilesLens
CameraProfilesLensPrettyName
CameraProfilesMake
CameraProfilesModel
CameraProfilesPerspectiveModel
CameraProfilesPerspectiveModelImageXCenter
CameraProfilesPerspectiveModelImageYCenter
CameraProfilesPerspectiveModelRadialDistortParam1
CameraProfilesPerspectiveModelRadialDistortParam2
CameraProfilesPerspectiveModelRadialDistortParam3
CameraProfilesPerspectiveModelScaleFactor
CameraProfilesPerspectiveModelVersion
CameraProfilesProfileName
CameraProfilesSensorFormatFactor
CameraProfilesUniqueCameraModel
CameraSerialNumber
CameraTrait
CameraVendorInfo
CameraVendorInfoManufacturer
CameraVendorInfoModel
CameraVendorInfoNotes
Cameras
Caption
CaptionWriter
CaptionsAuthorNames
CaptionsDateTimeStamps
CaptureSoftware
CatalogSets
Categories
Category
CellGlobalID
CellR
CellTowerID
Certificate
Channel
ChannelA-lang
ChannelChannel
ChannelSubchannel1
ChannelSubchannel2
ChannelSubchannel3
ChannelSubchannel4
ChildFontFiles
ChromaticAberrationB
ChromaticAberrationR
CircGradBasedCorrActive
CircGradBasedCorrAmount
CircGradBasedCorrBlacks2012
CircGradBasedCorrBrightness
CircGradBasedCorrClarity
CircGradBasedCorrClarity2012
CircGradBasedCorrContrast
CircGradBasedCorrContrast2012
CircGradBasedCorrCorrectionName
CircGradBasedCorrCorrectionSyncID
CircGradBasedCorrDefringe
CircGradBasedCorrDehaze
CircGradBasedCorrExposure
CircGradBasedCorrExposure2012
CircGradBasedCorrHighlights2012
CircGradBasedCorrHue
CircGradBasedCorrLuminanceNoise
CircGradBasedCorrMaskAlpha
CircGradBasedCorrMaskAngle
CircGradBasedCorrMaskBottom
CircGradBasedCorrMaskCenterValue
CircGradBasedCorrMaskCenterWeight
CircGradBasedCorrMaskDabs
CircGradBasedCorrMaskFeather
CircGradBasedCorrMaskFlipped
CircGradBasedCorrMaskFlow
CircGradBasedCorrMaskFullX
CircGradBasedCorrMaskFullY
CircGradBasedCorrMaskInputDigest
CircGradBasedCorrMaskLeft
CircGradBasedCorrMaskMaskActive
CircGradBasedCorrMaskMaskBlendMode
CircGradBasedCorrMaskMaskDigest
CircGradBasedCorrMaskMaskInverted
CircGradBasedCorrMaskMaskName
CircGradBasedCorrMaskMaskSubType
CircGradBasedCorrMaskMaskSyncID
CircGradBasedCorrMaskMaskVersion
CircGradBasedCorrMaskMasks
CircGradBasedCorrMaskMasksAlpha
CircGradBasedCorrMaskMasksAngle
CircGradBasedCorrMaskMasksBottom
CircGradBasedCorrMaskMasksCenterValue
CircGradBasedCorrMaskMasksCenterWeight
CircGradBasedCorrMaskMasksDabs
CircGradBasedCorrMaskMasksFeather
CircGradBasedCorrMaskMasksFlipped
CircGradBasedCorrMaskMasksFlow
CircGradBasedCorrMaskMasksFullX
CircGradBasedCorrMaskMasksFullY
CircGradBasedCorrMaskMasksInputDigest
CircGradBasedCorrMaskMasksLeft
CircGradBasedCorrMaskMasksMaskActive
CircGradBasedCorrMaskMasksMaskBlendMode
CircGradBasedCorrMaskMasksMaskDigest
CircGradBasedCorrMaskMasksMaskInverted
CircGradBasedCorrMaskMasksMaskName
CircGradBasedCorrMaskMasksMaskSubType
CircGradBasedCorrMaskMasksMaskSyncID
CircGradBasedCorrMaskMasksMaskVersion
CircGradBasedCorrMaskMasksMidpoint
CircGradBasedCorrMaskMasksOrigin
CircGradBasedCorrMaskMasksPerimeterValue
CircGradBasedCorrMaskMasksRadius
CircGradBasedCorrMaskMasksReferencePoint
CircGradBasedCorrMaskMasksRight
CircGradBasedCorrMaskMasksRoundness
CircGradBasedCorrMaskMasksSizeX
CircGradBasedCorrMaskMasksSizeY
CircGradBasedCorrMaskMasksTop
CircGradBasedCorrMaskMasksValue
CircGradBasedCorrMaskMasksVersion
CircGradBasedCorrMaskMasksWhat
CircGradBasedCorrMaskMasksWholeImageArea
CircGradBasedCorrMaskMasksX
CircGradBasedCorrMaskMasksY
CircGradBasedCorrMaskMasksZeroX
CircGradBasedCorrMaskMasksZeroY
CircGradBasedCorrMaskMidpoint
CircGradBasedCorrMaskOrigin
CircGradBasedCorrMaskPerimeterValue
CircGradBasedCorrMaskRadius
CircGradBasedCorrMaskRange
CircGradBasedCorrMaskRangeAreaModels
CircGradBasedCorrMaskRangeAreaModelsColorSampleInfo
CircGradBasedCorrMaskRangeAreaModelsComponents
CircGradBasedCorrMaskRangeColorAmount
CircGradBasedCorrMaskRangeDepthFeather
CircGradBasedCorrMaskRangeDepthMax
CircGradBasedCorrMaskRangeDepthMin
CircGradBasedCorrMaskRangeInvert
CircGradBasedCorrMaskRangeLumFeather
CircGradBasedCorrMaskRangeLumMax
CircGradBasedCorrMaskRangeLumMin
CircGradBasedCorrMaskRangeLumRange
CircGradBasedCorrMaskRangeLuminanceDepthSampleInfo
CircGradBasedCorrMaskRangeSampleType
CircGradBasedCorrMaskRangeType
CircGradBasedCorrMaskRangeVersion
CircGradBasedCorrMaskReferencePoint
CircGradBasedCorrMaskRight
CircGradBasedCorrMaskRoundness
CircGradBasedCorrMaskSizeX
CircGradBasedCorrMaskSizeY
CircGradBasedCorrMaskTop
CircGradBasedCorrMaskValue
CircGradBasedCorrMaskVersion
CircGradBasedCorrMaskWhat
CircGradBasedCorrMaskWholeImageArea
CircGradBasedCorrMaskX
CircGradBasedCorrMaskY
CircGradBasedCorrMaskZeroX
CircGradBasedCorrMaskZeroY
CircGradBasedCorrMasks
CircGradBasedCorrMoire
CircGradBasedCorrRangeMask
CircGradBasedCorrRangeMaskAreaModels
CircGradBasedCorrRangeMaskAreaModelsColorSampleInfo
CircGradBasedCorrRangeMaskAreaModelsComponents
CircGradBasedCorrRangeMaskColorAmount
CircGradBasedCorrRangeMaskDepthFeather
CircGradBasedCorrRangeMaskDepthMax
CircGradBasedCorrRangeMaskDepthMin
CircGradBasedCorrRangeMaskInvert
CircGradBasedCorrRangeMaskLumFeather
CircGradBasedCorrRangeMaskLumMax
CircGradBasedCorrRangeMaskLumMin
CircGradBasedCorrRangeMaskLumRange
CircGradBasedCorrRangeMaskLuminanceDepthSampleInfo
CircGradBasedCorrRangeMaskSampleType
CircGradBasedCorrRangeMaskType
CircGradBasedCorrRangeMaskVersion
CircGradBasedCorrSaturation
CircGradBasedCorrShadows2012
CircGradBasedCorrSharpness
CircGradBasedCorrTemperature
CircGradBasedCorrTexture
CircGradBasedCorrTint
CircGradBasedCorrToningHue
CircGradBasedCorrToningSaturation
CircGradBasedCorrWhat
CircGradBasedCorrWhites2012
CircaDateCreated
CircularGradientBasedCorrections
City
Clarity
Clarity2012
Client
ClientName
ClipboardAspectRatio
ClipboardOrientation
Cluster
CollectionName
CollectionURI
Collections
Color
ColorClass
ColorGradeBlending
ColorGradeGlobalHue
ColorGradeGlobalLum
ColorGradeGlobalSat
ColorGradeHighlightLum
ColorGradeMidtoneHue
ColorGradeMidtoneLum
ColorGradeMidtoneSat
ColorGradeShadowLum
ColorLabel
ColorMode
ColorNoiseReduction
ColorNoiseReductionDetail
ColorNoiseReductionSmoothness
ColorTemperature
ColorantA
ColorantB
ColorantBlack
ColorantBlue
ColorantCyan
ColorantGray
ColorantGreen
ColorantL
ColorantMagenta
ColorantMode
ColorantRed
ColorantSwatchName
ColorantTint
ColorantType
ColorantYellow
Colorants
CompImageImagesPerSequence
CompImageMaxExposureAll
CompImageMaxExposureUsed
CompImageMinExposureAll
CompImageMinExposureUsed
CompImageNumSequences
CompImageSumExposureAll
CompImageSumExposureUsed
CompImageTotalExposurePeriod
CompImageValues
CompatibleVersion
ComplianceProfile
Composer
CompositeImage
CompositeImageCount
CompositeImageExposureTimes
Composition
Confidence
ConfidenceLevel
ConfidenceMime
ContactInfo
Container
ContainerDirectory
ContainerDirectoryItem
ContainerDirectoryItemDataURI
ContainerDirectoryItemLength
ContainerDirectoryItemMime
ContainerDirectoryItemPadding
ContainerFormat
ContainerFormatIdentifier
ContainerFormatName
ContentType
Contrast
Contrast2012
ContributedMedia
ContributedMediaDuration
ContributedMediaDurationScale
ContributedMediaDurationValue
ContributedMediaManaged
ContributedMediaPath
ContributedMediaStartTime
ContributedMediaStartTimeScale
ContributedMediaStartTimeValue
ContributedMediaTrack
ContributedMediaWebStatement
Contributor
ContributorIdentifier
ContributorName
ContributorRole
ControlledVocabularyTerm
ConvertToGrayscale
Converter
CookingEquipment
CookingMethod
Copyright
CopyrightOwner
CopyrightOwnerID
CopyrightOwnerImageID
CopyrightOwnerName
CopyrightRegistrationNumber
CopyrightStatus
CopyrightYear
CorporateEntity
CorrectionAlreadyApplied
Country
CountryCode
Course
CoverDate
CoverDisplayDate
Coverage
CreateDate
CreationDate
Creator
CreatorAddress
CreatorAppID
CreatorCity
CreatorContactInfo
CreatorCountry
CreatorIdentifier
CreatorIdentity
CreatorName
CreatorOpenWithUIOptions
CreatorPostalCode
CreatorRegion
CreatorRole
CreatorTool
CreatorWorkEmail
CreatorWorkTelephone
CreatorWorkURL
Credit
CreditLine
CreditLineReq
CreditLineRequired
CropAngle
CropBottom
CropConstrainToWarp
CropH
CropHeight
CropLeft
CropRight
CropTop
CropUnit
CropUnits
CropW
CropWidth
CropX
CropY
CroppedAreaImageHeightPixels
CroppedAreaImageWidthPixels
CroppedAreaLeftPixels
CroppedAreaTopPixels
Cuisine
Curve0x
Curve0y
Curve1x
Curve1y
Curve2x
Curve2y
Curve3x
Curve3y
Curve4x
Curve4y
Custom1
Custom10
Custom2
Custom3
Custom4
Custom5
Custom6
Custom7
Custom8
Custom9
CustomRendered
DCContinent
DCCoordinatePrecision
DCCoordinateUncertaintyInMeters
DCCountry
DCCountryCode
DCCounty
DCDecimalLatitude
DCDecimalLongitude
DCEvent
DCFootprintSRS
DCFootprintSpatialFit
DCFootprintWKT
DCGeodeticDatum
DCGeoreferenceProtocol
DCGeoreferenceRemarks
DCGeoreferenceSources
DCGeoreferenceVerificationStatus
DCGeoreferencedBy
DCGeoreferencedDate
DCHigherGeography
DCHigherGeographyID
DCIsland
DCIslandGroup
DCLocality
DCLocationAccordingTo
DCLocationID
DCLocationRemarks
DCMaximumDepthInMeters
DCMaximumDistanceAboveSurfaceInMeters
DCMaximumElevationInMeters
DCMinimumDepthInMeters
DCMinimumDistanceAboveSurfaceInMeters
DCMinimumElevationInMeters
DCMunicipality
DCPointRadiusSpatialFit
DCStateProvince
DCTermsLocation
DCVerbatimCoordinateSystem
DCVerbatimCoordinates
DCVerbatimDepth
DCVerbatimElevation
DCVerbatimLatitude
DCVerbatimLocality
DCVerbatimLongitude
DCVerbatimSRS
DCVerticalDatum
DCWaterBody
DMComment
DNGAdobeData
DNGIgnoreSidecars
DNGLensInfo
DNGPrivateData
DOI
DPP
DataMining
DataOnScreen
DataOnScreenRegion
DataOnScreenRegionD
DataOnScreenRegionH
DataOnScreenRegionText
DataOnScreenRegionUnit
DataOnScreenRegionW
DataOnScreenRegionX
DataOnScreenRegionY
Date
DateAcquired
DateCreated
DateIdentified
DateRecieved
DateTime
DateTimeDigitized
DateTimeOriginal
DefaultAutoGray
DefaultAutoTone
DefaultsSpecificToISO
DefaultsSpecificToSerial
Defringe
DefringeGreenAmount
DefringeGreenHueHi
DefringeGreenHueLo
DefringePurpleAmount
DefringePurpleHueHi
DefringePurpleHueLo
Dehaze
DeprecatedOn
DepthBasedCorrCorrectionActive
DepthBasedCorrCorrectionAmount
DepthBasedCorrCorrectionSyncID
DepthBasedCorrLocalCorrectedDepth
DepthBasedCorrLocalCurveRefineSaturation
DepthBasedCorrMask
DepthBasedCorrMaskAlpha
DepthBasedCorrMaskAngle
DepthBasedCorrMaskBottom
DepthBasedCorrMaskCenterValue
DepthBasedCorrMaskCenterWeight
DepthBasedCorrMaskDabs
DepthBasedCorrMaskFeather
DepthBasedCorrMaskFlipped
DepthBasedCorrMaskFlow
DepthBasedCorrMaskFullX
DepthBasedCorrMaskFullY
DepthBasedCorrMaskInputDigest
DepthBasedCorrMaskLeft
DepthBasedCorrMaskMaskActive
DepthBasedCorrMaskMaskBlendMode
DepthBasedCorrMaskMaskDigest
DepthBasedCorrMaskMaskInverted
DepthBasedCorrMaskMaskName
DepthBasedCorrMaskMaskSubType
DepthBasedCorrMaskMaskSyncID
DepthBasedCorrMaskMaskVersion
DepthBasedCorrMaskMasks
DepthBasedCorrMaskMasksAlpha
DepthBasedCorrMaskMasksAngle
DepthBasedCorrMaskMasksBottom
DepthBasedCorrMaskMasksCenterValue
DepthBasedCorrMaskMasksCenterWeight
DepthBasedCorrMaskMasksDabs
DepthBasedCorrMaskMasksFeather
DepthBasedCorrMaskMasksFlipped
DepthBasedCorrMaskMasksFlow
DepthBasedCorrMaskMasksFullX
DepthBasedCorrMaskMasksFullY
DepthBasedCorrMaskMasksInputDigest
DepthBasedCorrMaskMasksLeft
DepthBasedCorrMaskMasksMaskActive
DepthBasedCorrMaskMasksMaskBlendMode
DepthBasedCorrMaskMasksMaskDigest
DepthBasedCorrMaskMasksMaskInverted
DepthBasedCorrMaskMasksMaskName
DepthBasedCorrMaskMasksMaskSubType
DepthBasedCorrMaskMasksMaskSyncID
DepthBasedCorrMaskMasksMaskVersion
DepthBasedCorrMaskMasksMidpoint
DepthBasedCorrMaskMasksOrigin
DepthBasedCorrMaskMasksPerimeterValue
DepthBasedCorrMaskMasksRadius
DepthBasedCorrMaskMasksReferencePoint
DepthBasedCorrMaskMasksRight
DepthBasedCorrMaskMasksRoundness
DepthBasedCorrMaskMasksSizeX
DepthBasedCorrMaskMasksSizeY
DepthBasedCorrMaskMasksTop
DepthBasedCorrMaskMasksValue
DepthBasedCorrMaskMasksVersion
DepthBasedCorrMaskMasksWhat
DepthBasedCorrMaskMasksWholeImageArea
DepthBasedCorrMaskMasksX
DepthBasedCorrMaskMasksY
DepthBasedCorrMaskMasksZeroX
DepthBasedCorrMaskMasksZeroY
DepthBasedCorrMaskMidpoint
DepthBasedCorrMaskOrigin
DepthBasedCorrMaskPerimeterValue
DepthBasedCorrMaskRadius
DepthBasedCorrMaskRange
DepthBasedCorrMaskRangeAreaModels
DepthBasedCorrMaskRangeAreaModelsColorSampleInfo
DepthBasedCorrMaskRangeAreaModelsComponents
DepthBasedCorrMaskRangeColorAmount
DepthBasedCorrMaskRangeDepthFeather
DepthBasedCorrMaskRangeDepthMax
DepthBasedCorrMaskRangeDepthMin
DepthBasedCorrMaskRangeInvert
DepthBasedCorrMaskRangeLumFeather
DepthBasedCorrMaskRangeLumMax
DepthBasedCorrMaskRangeLumMin
DepthBasedCorrMaskRangeLumRange
DepthBasedCorrMaskRangeLuminanceDepthSampleInfo
DepthBasedCorrMaskRangeSampleType
DepthBasedCorrMaskRangeType
DepthBasedCorrMaskRangeVersion
DepthBasedCorrMaskReferencePoint
DepthBasedCorrMaskRight
DepthBasedCorrMaskRoundness
DepthBasedCorrMaskSizeX
DepthBasedCorrMaskSizeY
DepthBasedCorrMaskTop
DepthBasedCorrMaskValue
DepthBasedCorrMaskVersion
DepthBasedCorrMaskWhat
DepthBasedCorrMaskWholeImageArea
DepthBasedCorrMaskX
DepthBasedCorrMaskY
DepthBasedCorrMaskZeroX
DepthBasedCorrMaskZeroY
DepthBasedCorrWhat
DepthBasedCorrections
DepthImage
DepthMapInfo
DepthMapInfoBaseHighlightGuideInputDigest
DepthMapInfoBaseHighlightGuideTable
DepthMapInfoBaseHighlightGuideVersion
DepthMapInfoBaseLayeredDepthInputDigest
DepthMapInfoBaseLayeredDepthTable
DepthMapInfoBaseLayeredDepthVersion
DepthMapInfoBaseRawDepthInputDigest
DepthMapInfoBaseRawDepthTable
DepthMapInfoBaseRawDepthVersion
DepthMapInfoDepthSource
DerivedFrom
DerivedFromAlternatePaths
DerivedFromDocumentID
DerivedFromFilePath
DerivedFromFromPart
DerivedFromInstanceID
DerivedFromLastModifyDate
DerivedFromLastURL
DerivedFromLinkCategory
DerivedFromLinkForm
DerivedFromManageTo
DerivedFromManageUI
DerivedFromManager
DerivedFromManagerVariant
DerivedFromMaskMarkers
DerivedFromOriginalDocumentID
DerivedFromPartMapping
DerivedFromPlacedResolutionUnit
DerivedFromPlacedXResolution
DerivedFromPlacedYResolution
DerivedFromRenditionClass
DerivedFromRenditionParams
DerivedFromToPart
DerivedFromVersionID
Description
Device
DeviceSettingDescription
DeviceSettingDescriptionColumns
DeviceSettingDescriptionRows
DeviceSettingDescriptionSettings
DewarpData
DewarpFlag
DietaryNeeds
DigitalImageGUID
DigitalSourceFileType
DigitalSourceType
DigitalZoomRatio
Director
DirectorPhotography
DisableAutoCreation
DiscNumber
DishType
DisplayName
DistortionCorrectionAlreadyApplied
Distributor
DistributorProductID
DocumentAncestors
DocumentID
DocumentName
Dopesheet
DopesheetLink
DopesheetLinkLink
DopesheetLinkLinkQualifier
Duration
DurationScale
DurationValue
EIssn
EXIF
EarliestAgeOrLowestStage
EarliestEonOrLowestEonothem
EarliestEpochOrLowestSeries
EarliestEraOrLowestErathem
EarliestPeriodOrLowestSystem
EarthPos
EarthPosAltitude
EarthPosLatitude
EarthPosLongitude
EarthPosRotationW
EarthPosRotationX
EarthPosRotationY
EarthPosRotationZ
EarthPosTimestamp
EditStatus
Edition
EmbargoDate
EmbdEncRightsExpr
EmbeddedEncodedRightsExpr
EmbeddedEncodedRightsExprLangID
EmbeddedEncodedRightsExprType
EmbeddedXMPDigest
EndUser
EndUserID
EndUserName
EndingPage
Engineer
EnhanceDenoiseAlreadyApplied
EnhanceDenoiseLumaAmount
EnhanceDenoiseVersion
EnhanceDetailsAlreadyApplied
EnhanceDetailsVersion
EnhanceSuperResolutionAlreadyApplied
EnhanceSuperResolutionScale
EnhanceSuperResolutionVersion
Episode
EpisodeIdentifier
EpisodeName
EpisodeNumber
EquipmentInstitution
EquipmentManufacturer
Event
EventAlias
EventDate
EventDay
EventEarliestDate
EventEnd
EventEndDayOfYear
EventFieldNotes
EventFieldNumber
EventHabitat
EventID
EventLatestDate
EventMonth
EventParentEventID
EventRemarks
EventSampleSizeUnit
EventSampleSizeValue
EventSamplingEffort
EventSamplingProtocol
EventStart
EventStartDayOfYear
EventSubtype
EventTime
EventType
EventVerbatimEventDate
EventYear
ExclusiveCoverage
ExclusivityEndDate
ExpirationDate
Exposure
Exposure2012
ExposureCompensation
ExposureIndex
ExposureLockUsed
ExposureMode
ExposureProgram
ExposureTime
ExtDescrAccessibility
ExternalMetadataLink
FFID
FNumber
FaceBalanceOrigI
FaceBalanceOrigQ
FaceBalanceStrength
FaceBalanceWarmth
FaceID
Far
FeedIdentifier
Field
FileDataRate
FileNameAsDelivered
FileSource
FillLight
Firmware
FirstPhotoDate
FirstPublicationDate
FixtureIdentifier
Flash
FlashCompensation
FlashEnergy
FlashFired
FlashFunction
FlashManufacturer
FlashMode
FlashModel
FlashRedEyeMode
FlashReturn
FlightPitchDegree
FlightRollDegree
FlightXSpeed
FlightYSpeed
FlightYawDegree
FlightZSpeed
FocalDistance
FocalLength
FocalLength35mm
FocalLengthIn35mmFormat
FocalPlaneResolutionUnit
FocalPlaneXResolution
FocalPlaneYResolution
FocalPointX
FocalPointY
FontComposite
FontFace
FontFamily
FontFileName
FontName
FontType
FontVersion
Fonts
Format
ForwardLock
FossilSpecimen
FossilSpecimenMaterialSampleID
FrameRate
Framing
FullPanoHeightPixels
FullPanoWidthPixels
GDALMetadata
GDALNoData
GIFTFtpPriority
GPSAltitude
GPSAltitudeRef
GPSAreaInformation
GPSDOP
GPSDateStamp
GPSDateTime
GPSDestBearing
GPSDestBearingRef
GPSDestDistance
GPSDestDistanceRef
GPSDestLatitude
GPSDestLatitudeRef
GPSDestLongitude
GPSDestLongitudeRef
GPSDifferential
GPSHPositioningError
GPSImgDirection
GPSImgDirectionRef
GPSLatitude
GPSLatitudeRef
GPSLongitude
GPSLongitudeRef
GPSLongtitude
GPSMapDatum
GPSMeasureMode
GPSProcessingMethod
GPSSatellites
GPSSpeed
GPSSpeedRef
GPSStatus
GPSTimeStamp
GPSTrack
GPSTrackRef
GPSVersionID
GainControl
GainMapMax
GainMapMin
Gamma
Genre
GenreCvId
GenreCvTermId
GenreCvTermName
GenreCvTermRefinedAbout
GeoTiffAsciiParams
GeoTiffDirectory
GeoTiffDoubleParams
Geography
GeologicalContext
GeologicalContextBed
GeologicalContextFormation
GeologicalContextGroup
GeologicalContextID
GeologicalContextMember
GimbalPitchDegree
GimbalReverse
GimbalRollDegree
GimbalYawDegree
Good
GooglePlusUploadCode
GradientBasedCorrActive
GradientBasedCorrAmount
GradientBasedCorrBlacks2012
GradientBasedCorrBrightness
GradientBasedCorrClarity
GradientBasedCorrClarity2012
GradientBasedCorrContrast
GradientBasedCorrContrast2012
GradientBasedCorrCorrectionName
GradientBasedCorrCorrectionSyncID
GradientBasedCorrDefringe
GradientBasedCorrDehaze
GradientBasedCorrExposure
GradientBasedCorrExposure2012
GradientBasedCorrHighlights2012
GradientBasedCorrHue
GradientBasedCorrLuminanceNoise
GradientBasedCorrMaskAlpha
GradientBasedCorrMaskAngle
GradientBasedCorrMaskBottom
GradientBasedCorrMaskCenterValue
GradientBasedCorrMaskCenterWeight
GradientBasedCorrMaskDabs
GradientBasedCorrMaskFeather
GradientBasedCorrMaskFlipped
GradientBasedCorrMaskFlow
GradientBasedCorrMaskFullX
GradientBasedCorrMaskFullY
GradientBasedCorrMaskInputDigest
GradientBasedCorrMaskLeft
GradientBasedCorrMaskMaskActive
GradientBasedCorrMaskMaskBlendMode
GradientBasedCorrMaskMaskDigest
GradientBasedCorrMaskMaskInverted
GradientBasedCorrMaskMaskName
GradientBasedCorrMaskMaskSubType
GradientBasedCorrMaskMaskSyncID
GradientBasedCorrMaskMaskVersion
GradientBasedCorrMaskMasks
GradientBasedCorrMaskMasksAlpha
GradientBasedCorrMaskMasksAngle
GradientBasedCorrMaskMasksBottom
GradientBasedCorrMaskMasksCenterValue
GradientBasedCorrMaskMasksCenterWeight
GradientBasedCorrMaskMasksDabs
GradientBasedCorrMaskMasksFeather
GradientBasedCorrMaskMasksFlipped
GradientBasedCorrMaskMasksFlow
GradientBasedCorrMaskMasksFullX
GradientBasedCorrMaskMasksFullY
GradientBasedCorrMaskMasksInputDigest
GradientBasedCorrMaskMasksLeft
GradientBasedCorrMaskMasksMaskActive
GradientBasedCorrMaskMasksMaskBlendMode
GradientBasedCorrMaskMasksMaskDigest
GradientBasedCorrMaskMasksMaskInverted
GradientBasedCorrMaskMasksMaskName
GradientBasedCorrMaskMasksMaskSubType
GradientBasedCorrMaskMasksMaskSyncID
GradientBasedCorrMaskMasksMaskVersion
GradientBasedCorrMaskMasksMidpoint
GradientBasedCorrMaskMasksOrigin
GradientBasedCorrMaskMasksPerimeterValue
GradientBasedCorrMaskMasksRadius
GradientBasedCorrMaskMasksReferencePoint
GradientBasedCorrMaskMasksRight
GradientBasedCorrMaskMasksRoundness
GradientBasedCorrMaskMasksSizeX
GradientBasedCorrMaskMasksSizeY
GradientBasedCorrMaskMasksTop
GradientBasedCorrMaskMasksValue
GradientBasedCorrMaskMasksVersion
GradientBasedCorrMaskMasksWhat
GradientBasedCorrMaskMasksWholeImageArea
GradientBasedCorrMaskMasksX
GradientBasedCorrMaskMasksY
GradientBasedCorrMaskMasksZeroX
GradientBasedCorrMaskMasksZeroY
GradientBasedCorrMaskMidpoint
GradientBasedCorrMaskOrigin
GradientBasedCorrMaskPerimeterValue
GradientBasedCorrMaskRadius
GradientBasedCorrMaskRange
GradientBasedCorrMaskRangeAreaModels
GradientBasedCorrMaskRangeAreaModelsColorSampleInfo
GradientBasedCorrMaskRangeAreaModelsComponents
GradientBasedCorrMaskRangeColorAmount
GradientBasedCorrMaskRangeDepthFeather
GradientBasedCorrMaskRangeDepthMax
GradientBasedCorrMaskRangeDepthMin
GradientBasedCorrMaskRangeInvert
GradientBasedCorrMaskRangeLumFeather
GradientBasedCorrMaskRangeLumMax
GradientBasedCorrMaskRangeLumMin
GradientBasedCorrMaskRangeLumRange
GradientBasedCorrMaskRangeLuminanceDepthSampleInfo
GradientBasedCorrMaskRangeSampleType
GradientBasedCorrMaskRangeType
GradientBasedCorrMaskRangeVersion
GradientBasedCorrMaskReferencePoint
GradientBasedCorrMaskRight
GradientBasedCorrMaskRoundness
GradientBasedCorrMaskSizeX
GradientBasedCorrMaskSizeY
GradientBasedCorrMaskTop
GradientBasedCorrMaskValue
GradientBasedCorrMaskVersion
GradientBasedCorrMaskWhat
GradientBasedCorrMaskWholeImageArea
GradientBasedCorrMaskX
GradientBasedCorrMaskY
GradientBasedCorrMaskZeroX
GradientBasedCorrMaskZeroY
GradientBasedCorrMasks
GradientBasedCorrMoire
GradientBasedCorrRangeMask
GradientBasedCorrRangeMaskAreaModels
GradientBasedCorrRangeMaskAreaModelsColorSampleInfo
GradientBasedCorrRangeMaskAreaModelsComponents
GradientBasedCorrRangeMaskColorAmount
GradientBasedCorrRangeMaskDepthFeather
GradientBasedCorrRangeMaskDepthMax
GradientBasedCorrRangeMaskDepthMin
GradientBasedCorrRangeMaskInvert
GradientBasedCorrRangeMaskLumFeather
GradientBasedCorrRangeMaskLumMax
GradientBasedCorrRangeMaskLumMin
GradientBasedCorrRangeMaskLumRange
GradientBasedCorrRangeMaskLuminanceDepthSampleInfo
GradientBasedCorrRangeMaskSampleType
GradientBasedCorrRangeMaskType
GradientBasedCorrRangeMaskVersion
GradientBasedCorrSaturation
GradientBasedCorrShadows2012
GradientBasedCorrSharpness
GradientBasedCorrTemperature
GradientBasedCorrTexture
GradientBasedCorrTint
GradientBasedCorrToningHue
GradientBasedCorrToningSaturation
GradientBasedCorrWhat
GradientBasedCorrWhites2012
GradientBasedCorrections
GrainAmount
GrainFrequency
GrainSeed
GrainSize
GrayMixerAqua
GrayMixerBlue
GrayMixerGreen
GrayMixerMagenta
GrayMixerOrange
GrayMixerPurple
GrayMixerRed
GrayMixerYellow
GrayResponseUnit
GreenHue
GreenSaturation
Group
HDRCapacityMax
HDRCapacityMin
HDREditMode
HDRPMakerNote
HalftoneHints
HasAlternative
HasCorrection
HasCorrectionA-lang
HasCorrectionA-platform
HasCorrectionText
HasCrop
HasExtendedXMP
HasSettings
HasTranslation
HasVisibleOverprint
HasVisibleTransparency
HdrPlusMakernote
Headline
HierarchicalKeywords
HierarchicalKeywords1
HierarchicalKeywords1Applied
HierarchicalKeywords1Children
HierarchicalKeywords2
HierarchicalKeywords2Applied
HierarchicalKeywords2Children
HierarchicalKeywords3
HierarchicalKeywords3Applied
HierarchicalKeywords3Children
HierarchicalKeywords4
HierarchicalKeywords4Applied
HierarchicalKeywords4Children
HierarchicalKeywords5
HierarchicalKeywords5Applied
HierarchicalKeywords5Children
HierarchicalKeywords6
HierarchicalKeywords6Applied
HierarchicalSubject
HighISOMultiplierBlue
HighISOMultiplierGreen
HighISOMultiplierRed
HighestBiostratigraphicZone
Highlight2012
HighlightRecovery
Highlights
Highlights2012
History
HistoryAction
HistoryChanged
HistoryInstanceID
HistoryParameters
HistorySoftwareAgent
HistoryWhen
HostComputer
HueAdjustmentAqua
HueAdjustmentBlue
HueAdjustmentGreen
HueAdjustmentMagenta
HueAdjustmentOrange
HueAdjustmentPurple
HueAdjustmentRed
HueAdjustmentYellow
HumanObservation
HumanObservationDay
HumanObservationEarliestDate
HumanObservationEndDayOfYear
HumanObservationEventDate
HumanObservationEventID
HumanObservationEventRemarks
HumanObservationEventTime
HumanObservationFieldNotes
HumanObservationFieldNumber
HumanObservationHabitat
HumanObservationLatestDate
HumanObservationMonth
HumanObservationParentEventID
HumanObservationSampleSizeUnit
HumanObservationSampleSizeValue
HumanObservationSamplingEffort
HumanObservationSamplingProtocol
HumanObservationStartDayOfYear
HumanObservationVerbatimEventDate
HumanObservationYear
Humidity
ICCProfileName
IPTCLastEdited
ISBN
ISO
ISOSpeed
ISOSpeedLatitudeyyy
ISOSpeedLatitudezzz
ISSN
Identification
IdentificationID
IdentificationQualifier
IdentificationReferences
IdentificationRemarks
IdentificationVerificationStatus
IdentifiedBy
IdentifiedByID
Identifier
ImageAlterationConstraints
ImageCreator
ImageCreatorID
ImageCreatorImageID
ImageCreatorName
ImageData
ImageDescription
ImageDuplicationConstraints
ImageEditingSoftware
ImageEditor
ImageFileConstraints
ImageFileFormatAsDelivered
ImageFileSizeAsDelivered
ImageHistory
ImageMimeType
ImageNumber
ImageRank
ImageRef
ImageRegion
ImageRegionBoundary
ImageRegionBoundaryH
ImageRegionBoundaryRx
ImageRegionBoundaryShape
ImageRegionBoundaryUnit
ImageRegionBoundaryVertices
ImageRegionBoundaryVerticesX
ImageRegionBoundaryVerticesY
ImageRegionBoundaryW
ImageRegionBoundaryX
ImageRegionBoundaryY
ImageRegionCtype
ImageRegionCtypeIdentifier
ImageRegionCtypeName
ImageRegionID
ImageRegionName
ImageRegionRole
ImageRegionRoleIdentifier
ImageRegionRoleName
ImageSequenceInfo
ImageSizeRestriction
ImageSupplier
ImageSupplierID
ImageSupplierImageID
ImageSupplierName
ImageType
ImageUniqueID
IncrementalTemperature
IncrementalTint
Industry
IngredientExclusion
Ingredients
IngredientsAlternatePaths
IngredientsDocumentID
IngredientsFilePath
IngredientsFromPart
IngredientsInstanceID
IngredientsLastModifyDate
IngredientsLastURL
IngredientsLinkCategory
IngredientsLinkForm
IngredientsManageTo
IngredientsManageUI
IngredientsManager
IngredientsManagerVariant
IngredientsMaskMarkers
IngredientsOriginalDocumentID
IngredientsPartMapping
IngredientsPlacedResolutionUnit
IngredientsPlacedXResolution
IngredientsPlacedYResolution
IngredientsRenditionClass
IngredientsRenditionParams
IngredientsToPart
IngredientsVersionID
InitialCameraDolly
InitialHorizontalFOVDegrees
InitialVerticalFOVDegrees
InitialViewHeadingDegrees
InitialViewPitchDegrees
InitialViewRollDegrees
InkSet
InstanceID
Instructions
Instrument
IntellectualGenre
IntergraphMatrix
IntroTime
IntroTimeScale
IntroTimeValue
IsAlternativeOf
IsCorrectionOf
IsMergedHDR
IsMergedPanorama
IsTranslationOf
IssueIdentifier
IssueName
IssueTeaser
IssueType
ItemSubType
JPEGHandling
JobName
JobRef
JobRefId
JobRefName
JobRefUrl
JobStatus
JpgFromRaw
JpgFromRawLength
JpgFromRawStart
Jurisdiction
Key
Keyword
KeywordInfo
Keywords
KillDate
KillDateA-platform
KillDateDate
Label
LabelName1
LabelName2
LabelName3
LabelName4
LabelName5
LabelName6
Language
LargestValidInteriorRectHeight
LargestValidInteriorRectLeft
LargestValidInteriorRectTop
LargestValidInteriorRectWidth
LastKeywordIPTC
LastKeywordXMP
LastPhotoDate
LastURL
LateralChromaticAberrationCorrectionAlreadyApplied
LatestAgeOrHighestStage
LatestEonOrHighestEonothem
LatestEpochOrHighestSeries
LatestEraOrHighestErathem
LatestPeriodOrHighestSystem
Latitude
LegacyIPTCDigest
LegalCode
Lens
LensBlur
LensBlurActive
LensBlurAmount
LensBlurBokehAspect
LensBlurBokehRotation
LensBlurBokehShape
LensBlurBokehShapeDetail
LensBlurCatEyeAmount
LensBlurCatEyeScale
LensBlurFocalRange
LensBlurFocalRangeSource
LensBlurHighlightsBoost
LensBlurHighlightsThreshold
LensBlurSampledArea
LensBlurSampledRange
LensBlurSphericalAberration
LensBlurSubjectRange
LensBlurVersion
LensCorrectionSettings
LensDistortInfo
LensID
LensInfo
LensMake
LensManualDistortionAmount
LensManufacturer
LensModel
LensProfileChromaticAberrationScale
LensProfileDigest
LensProfileDistortionScale
LensProfileEnable
LensProfileFilename
LensProfileIsEmbedded
LensProfileMatchKeyCameraModelName
LensProfileMatchKeyExifMake
LensProfileMatchKeyExifModel
LensProfileMatchKeyIsRaw
LensProfileMatchKeyLensID
LensProfileMatchKeyLensInfo
LensProfileMatchKeyLensName
LensProfileMatchKeySensorFormatFactor
LensProfileName
LensProfileSetup
LensProfileVignettingScale
LensSerialNumber
License
LicenseEndDate
LicenseID
LicenseStartDate
LicenseTransactionDate
LicenseType
Licensee
LicenseeID
LicenseeImageID
LicenseeImageNotes
LicenseeName
LicenseeProjectReference
LicenseeTransactionID
Licensor
LicensorCity
LicensorCountry
LicensorEmail
LicensorExtendedAddress
LicensorID
LicensorImageID
LicensorName
LicensorNotes
LicensorPostalCode
LicensorRegion
LicensorStreetAddress
LicensorTelephone1
LicensorTelephone2
LicensorTelephoneType1
LicensorTelephoneType2
LicensorTransactionID
LicensorURL
LightSource
LinearityLimitBlue
LinearityLimitGreen
LinearityLimitRed
Link
LinkedEncRightsExpr
LinkedEncodedRightsExpr
LinkedEncodedRightsExprLangID
LinkedEncodedRightsExprType
LithostratigraphicTerms
LivingSpecimen
LivingSpecimenMaterialSampleID
LocalizedCameraModel
Location
LocationAreaCode
LocationCreated
LocationCreatedCity
LocationCreatedCountryCode
LocationCreatedCountryName
LocationCreatedGPSAltitude
LocationCreatedGPSLatitude
LocationCreatedGPSLongitude
LocationCreatedIdentifier
LocationCreatedLocationId
LocationCreatedLocationName
LocationCreatedProvinceState
LocationCreatedSublocation
LocationCreatedWorldRegion
LocationShown
LocationShownCity
LocationShownCountryCode
LocationShownCountryName
LocationShownGPSAltitude
LocationShownGPSLatitude
LocationShownGPSLongitude
LocationShownIdentifier
LocationShownLocationId
LocationShownLocationName
LocationShownProvinceState
LocationShownSublocation
LocationShownWorldRegion
LogComment
Longitude
Look
LookAmount
LookCluster
LookCopyright
LookGroup
LookName
LookParameters
LookParametersCameraProfile
LookParametersClarity2012
LookParametersConvertToGrayscale
LookParametersHighlights2012
LookParametersLookTable
LookParametersProcessVersion
LookParametersShadows2012
LookParametersToneCurvePV2012
LookParametersToneCurvePV2012Blue
LookParametersToneCurvePV2012Green
LookParametersToneCurvePV2012Red
LookParametersVersion
LookSupportsAmount
LookSupportsMonochrome
LookSupportsOutputReferred
LookUUID
Loop
LowestBiostratigraphicZone
LuminanceAdjustmentAqua
LuminanceAdjustmentBlue
LuminanceAdjustmentGreen
LuminanceAdjustmentMagenta
LuminanceAdjustmentOrange
LuminanceAdjustmentPurple
LuminanceAdjustmentRed
LuminanceAdjustmentYellow
LuminanceNoiseReductionContrast
LuminanceNoiseReductionDetail
LuminanceSmoothing
Lyrics
MacAtom
MacAtomApplicationCode
MacAtomInvocationAppleEvent
MacAtomPosixProjectPath
MachineObservation
MachineObservationDay
MachineObservationEarliestDate
MachineObservationEndDayOfYear
MachineObservationEventDate
MachineObservationEventID
MachineObservationEventRemarks
MachineObservationEventTime
MachineObservationFieldNotes
MachineObservationFieldNumber
MachineObservationHabitat
MachineObservationLatestDate
MachineObservationMonth
MachineObservationParentEventID
MachineObservationSampleSizeUnit
MachineObservationSampleSizeValue
MachineObservationSamplingEffort
MachineObservationSamplingProtocol
MachineObservationStartDayOfYear
MachineObservationVerbatimEventDate
MachineObservationYear
MainIngredient
MajorVersion
Make
MakerNote
MakerNoteApple
MakerNoteCanon
MakerNoteCasio
MakerNoteCasio2
MakerNoteDJI
MakerNoteDJIInfo
MakerNoteFLIR
MakerNoteFujiFilm
MakerNoteGE
MakerNoteGE2
MakerNoteHP
MakerNoteHP2
MakerNoteHP4
MakerNoteHP6
MakerNoteHasselblad
MakerNoteISL
MakerNoteJVC
MakerNoteJVCText
MakerNoteKodak10
MakerNoteKodak11
MakerNoteKodak12
MakerNoteKodak1a
MakerNoteKodak1b
MakerNoteKodak2
MakerNoteKodak3
MakerNoteKodak4
MakerNoteKodak5
MakerNoteKodak6a
MakerNoteKodak6b
MakerNoteKodak7
MakerNoteKodak8a
MakerNoteKodak8b
MakerNoteKodak8c
MakerNoteKodak9
MakerNoteKodakUnknown
MakerNoteKyocera
MakerNoteLeica
MakerNoteLeica10
MakerNoteLeica2
MakerNoteLeica3
MakerNoteLeica4
MakerNoteLeica5
MakerNoteLeica6
MakerNoteLeica7
MakerNoteLeica8
MakerNoteLeica9
MakerNoteMinolta
MakerNoteMinolta2
MakerNoteMinolta3
MakerNoteMotorola
MakerNoteNikon
MakerNoteNikon2
MakerNoteNikon3
MakerNoteNintendo
MakerNoteOlympus
MakerNoteOlympus2
MakerNoteOlympus3
MakerNotePanasonic
MakerNotePanasonic2
MakerNotePanasonic3
MakerNotePentax
MakerNotePentax2
MakerNotePentax3
MakerNotePentax4
MakerNotePentax5
MakerNotePentax6
MakerNotePhaseOne
MakerNoteReconyx
MakerNoteReconyx2
MakerNoteReconyx3
MakerNoteRicoh
MakerNoteRicoh2
MakerNoteRicohPentax
MakerNoteRicohText
MakerNoteSafety
MakerNoteSamsung1a
MakerNoteSamsung1b
MakerNoteSamsung2
MakerNoteSanyo
MakerNoteSanyoC4
MakerNoteSanyoPatch
MakerNoteSigma
MakerNoteSony
MakerNoteSony2
MakerNoteSony3
MakerNoteSony4
MakerNoteSony5
MakerNoteSonyEricsson
MakerNoteSonySRF
MakerNoteUnknown
MakerNoteUnknownBinary
MakerNoteUnknownText
ManageTo
ManageUI
ManagedFrom
ManagedFromAlternatePaths
ManagedFromDocumentID
ManagedFromFilePath
ManagedFromFromPart
ManagedFromInstanceID
ManagedFromLastModifyDate
ManagedFromLastURL
ManagedFromLinkCategory
ManagedFromLinkForm
ManagedFromManageTo
ManagedFromManageUI
ManagedFromManager
ManagedFromManagerVariant
ManagedFromMaskMarkers
ManagedFromOriginalDocumentID
ManagedFromPartMapping
ManagedFromPlacedResolutionUnit
ManagedFromPlacedXResolution
ManagedFromPlacedYResolution
ManagedFromRenditionClass
ManagedFromRenditionParams
ManagedFromToPart
ManagedFromVersionID
Manager
ManagerVariant
Manifest
ManifestLinkForm
ManifestPlacedResolutionUnit
ManifestPlacedXResolution
ManifestPlacedYResolution
ManifestReference
ManifestReferenceAlternatePaths
ManifestReferenceDocumentID
ManifestReferenceFilePath
ManifestReferenceFromPart
ManifestReferenceInstanceID
ManifestReferenceLastModifyDate
ManifestReferenceLastURL
ManifestReferenceLinkCategory
ManifestReferenceLinkForm
ManifestReferenceManageTo
ManifestReferenceManageUI
ManifestReferenceManager
ManifestReferenceManagerVariant
ManifestReferenceMaskMarkers
ManifestReferenceOriginalDocumentID
ManifestReferencePartMapping
ManifestReferencePlacedResolutionUnit
ManifestReferencePlacedXResolution
ManifestReferencePlacedYResolution
ManifestReferenceRenditionClass
ManifestReferenceRenditionParams
ManifestReferenceToPart
ManifestReferenceVersionID
Manufacturer
Marked
Markers
MarkersComment
MarkersCuePointParams
MarkersCuePointParamsKey
MarkersCuePointParamsValue
MarkersCuePointType
MarkersDuration
MarkersLocation
MarkersName
MarkersProbability
MarkersSpeaker
MarkersStartTime
MarkersTarget
MarkersType
MaskGroupBasedCorrActive
MaskGroupBasedCorrAmount
MaskGroupBasedCorrBlacks2012
MaskGroupBasedCorrBrightness
MaskGroupBasedCorrClarity
MaskGroupBasedCorrClarity2012
MaskGroupBasedCorrContrast
MaskGroupBasedCorrContrast2012
MaskGroupBasedCorrCorrectionName
MaskGroupBasedCorrCorrectionSyncID
MaskGroupBasedCorrDefringe
MaskGroupBasedCorrDehaze
MaskGroupBasedCorrExposure
MaskGroupBasedCorrExposure2012
MaskGroupBasedCorrHighlights2012
MaskGroupBasedCorrHue
MaskGroupBasedCorrLuminanceNoise
MaskGroupBasedCorrMask
MaskGroupBasedCorrMaskAlpha
MaskGroupBasedCorrMaskAngle
MaskGroupBasedCorrMaskBottom
MaskGroupBasedCorrMaskCenterValue
MaskGroupBasedCorrMaskCenterWeight
MaskGroupBasedCorrMaskDabs
MaskGroupBasedCorrMaskFeather
MaskGroupBasedCorrMaskFlipped
MaskGroupBasedCorrMaskFlow
MaskGroupBasedCorrMaskFullX
MaskGroupBasedCorrMaskFullY
MaskGroupBasedCorrMaskInputDigest
MaskGroupBasedCorrMaskLeft
MaskGroupBasedCorrMaskMaskActive
MaskGroupBasedCorrMaskMaskBlendMode
MaskGroupBasedCorrMaskMaskDigest
MaskGroupBasedCorrMaskMaskInverted
MaskGroupBasedCorrMaskMaskName
MaskGroupBasedCorrMaskMaskSubType
MaskGroupBasedCorrMaskMaskSyncID
MaskGroupBasedCorrMaskMaskVersion
MaskGroupBasedCorrMaskMasks
MaskGroupBasedCorrMaskMasksAlpha
MaskGroupBasedCorrMaskMasksAngle
MaskGroupBasedCorrMaskMasksBottom
MaskGroupBasedCorrMaskMasksCenterValue
MaskGroupBasedCorrMaskMasksCenterWeight
MaskGroupBasedCorrMaskMasksDabs
MaskGroupBasedCorrMaskMasksFeather
MaskGroupBasedCorrMaskMasksFlipped
MaskGroupBasedCorrMaskMasksFlow
MaskGroupBasedCorrMaskMasksFullX
MaskGroupBasedCorrMaskMasksFullY
MaskGroupBasedCorrMaskMasksInputDigest
MaskGroupBasedCorrMaskMasksLeft
MaskGroupBasedCorrMaskMasksMaskActive
MaskGroupBasedCorrMaskMasksMaskBlendMode
MaskGroupBasedCorrMaskMasksMaskDigest
MaskGroupBasedCorrMaskMasksMaskInverted
MaskGroupBasedCorrMaskMasksMaskName
MaskGroupBasedCorrMaskMasksMaskSubType
MaskGroupBasedCorrMaskMasksMaskSyncID
MaskGroupBasedCorrMaskMasksMaskVersion
MaskGroupBasedCorrMaskMasksMidpoint
MaskGroupBasedCorrMaskMasksOrigin
MaskGroupBasedCorrMaskMasksPerimeterValue
MaskGroupBasedCorrMaskMasksRadius
MaskGroupBasedCorrMaskMasksReferencePoint
MaskGroupBasedCorrMaskMasksRight
MaskGroupBasedCorrMaskMasksRoundness
MaskGroupBasedCorrMaskMasksSizeX
MaskGroupBasedCorrMaskMasksSizeY
MaskGroupBasedCorrMaskMasksTop
MaskGroupBasedCorrMaskMasksValue
MaskGroupBasedCorrMaskMasksVersion
MaskGroupBasedCorrMaskMasksWhat
MaskGroupBasedCorrMaskMasksWholeImageArea
MaskGroupBasedCorrMaskMasksX
MaskGroupBasedCorrMaskMasksY
MaskGroupBasedCorrMaskMasksZeroX
MaskGroupBasedCorrMaskMasksZeroY
MaskGroupBasedCorrMaskMidpoint
MaskGroupBasedCorrMaskOrigin
MaskGroupBasedCorrMaskPerimeterValue
MaskGroupBasedCorrMaskRadius
MaskGroupBasedCorrMaskRange
MaskGroupBasedCorrMaskRangeAreaModels
MaskGroupBasedCorrMaskRangeAreaModelsColorSampleInfo
MaskGroupBasedCorrMaskRangeAreaModelsComponents
MaskGroupBasedCorrMaskRangeColorAmount
MaskGroupBasedCorrMaskRangeDepthFeather
MaskGroupBasedCorrMaskRangeDepthMax
MaskGroupBasedCorrMaskRangeDepthMin
MaskGroupBasedCorrMaskRangeInvert
MaskGroupBasedCorrMaskRangeLumFeather
MaskGroupBasedCorrMaskRangeLumMax
MaskGroupBasedCorrMaskRangeLumMin
MaskGroupBasedCorrMaskRangeLumRange
MaskGroupBasedCorrMaskRangeLuminanceDepthSampleInfo
MaskGroupBasedCorrMaskRangeSampleType
MaskGroupBasedCorrMaskRangeType
MaskGroupBasedCorrMaskRangeVersion
MaskGroupBasedCorrMaskReferencePoint
MaskGroupBasedCorrMaskRight
MaskGroupBasedCorrMaskRoundness
MaskGroupBasedCorrMaskSizeX
MaskGroupBasedCorrMaskSizeY
MaskGroupBasedCorrMaskTop
MaskGroupBasedCorrMaskValue
MaskGroupBasedCorrMaskVersion
MaskGroupBasedCorrMaskWhat
MaskGroupBasedCorrMaskWholeImageArea
MaskGroupBasedCorrMaskX
MaskGroupBasedCorrMaskY
MaskGroupBasedCorrMaskZeroX
MaskGroupBasedCorrMaskZeroY
MaskGroupBasedCorrMoire
MaskGroupBasedCorrRangeMask
MaskGroupBasedCorrRangeMaskAreaModels
MaskGroupBasedCorrRangeMaskAreaModelsColorSampleInfo
MaskGroupBasedCorrRangeMaskAreaModelsComponents
MaskGroupBasedCorrRangeMaskColorAmount
MaskGroupBasedCorrRangeMaskDepthFeather
MaskGroupBasedCorrRangeMaskDepthMax
MaskGroupBasedCorrRangeMaskDepthMin
MaskGroupBasedCorrRangeMaskInvert
MaskGroupBasedCorrRangeMaskLumFeather
MaskGroupBasedCorrRangeMaskLumMax
MaskGroupBasedCorrRangeMaskLumMin
MaskGroupBasedCorrRangeMaskLumRange
MaskGroupBasedCorrRangeMaskLuminanceDepthSampleInfo
MaskGroupBasedCorrRangeMaskSampleType
MaskGroupBasedCorrRangeMaskType
MaskGroupBasedCorrRangeMaskVersion
MaskGroupBasedCorrSaturation
MaskGroupBasedCorrShadows2012
MaskGroupBasedCorrSharpness
MaskGroupBasedCorrTemperature
MaskGroupBasedCorrTexture
MaskGroupBasedCorrTint
MaskGroupBasedCorrToningHue
MaskGroupBasedCorrToningSaturation
MaskGroupBasedCorrWhat
MaskGroupBasedCorrWhites2012
MaskGroupBasedCorrections
MaterialSample
MaterialSampleID
MaxApertureValue
MaxAvailHeight
MaxAvailWidth
MaxPageSize
MaxPageSizeH
MaxPageSizeUnit
MaxPageSizeW
MaxSampleValue
MaxStorage
Meal
MeasureType
MeasurementAccuracy
MeasurementDeterminedBy
MeasurementDeterminedDate
MeasurementID
MeasurementMethod
MeasurementOrFact
MeasurementRemarks
MeasurementType
MeasurementUnit
MeasurementValue
MediaConstraints
MediaEventIdDate
MediaSummaryCode
MetadataAuthority
MetadataAuthorityIdentifier
MetadataAuthorityName
MetadataDate
MetadataEditingSoftware
MetadataLastEdited
MetadataLastEditor
MetadataLastEditorIdentifier
MetadataLastEditorName
MetadataModDate
MeteringMode
MicroVideo
MicroVideoOffset
MicroVideoPresentationTimestampUs
MicroVideoVersion
Mime
MinSampleValue
MinorModelAgeDisclosure
MinorVersion
MobileCountryCode
MobileNetworkCode
ModDate
Model
ModelAge
ModelReleaseID
ModelReleaseStatus
ModelTiePoint
ModelTransform
ModelYear
ModificationDate
ModifyDate
MoireFilter
MorePermissions
Multishot
NPages
Name
NationalCatalogNumber
NativeDigest
Near
NegativeCacheLargePreviewSize
NegativeCacheMaximumSize
NegativeCachePath
NeutralDensityFactor
Nickname
NoiseReductionParams
NormalizedCropCorners
Notes
Number
NumberOfBeats
OECFColumns
OECFNames
OECFRows
OECFValues
OS
Object
ObjectCycle
ObjectDescription
ObjectSubtype
ObjectType
Occurrence
OccurrenceAssociatedMedia
OccurrenceAssociatedOccurrences
OccurrenceAssociatedReferences
OccurrenceAssociatedSequences
OccurrenceAssociatedTaxa
OccurrenceBehavior
OccurrenceCatalogNumber
OccurrenceDegreeOfEstablishment
OccurrenceDetails
OccurrenceDisposition
OccurrenceEstablishmentMeans
OccurrenceGeoreferenceVerificationStatus
OccurrenceID
OccurrenceIndividualCount
OccurrenceIndividualID
OccurrenceLifeStage
OccurrenceOrganismQuantity
OccurrenceOrganismQuantityType
OccurrenceOtherCatalogNumbers
OccurrencePathway
OccurrencePreparations
OccurrencePreviousIdentifications
OccurrenceRecordNumber
OccurrenceRecordedBy
OccurrenceRecordedByID
OccurrenceRemarks
OccurrenceReproductiveCondition
OccurrenceSex
OccurrenceStatus
OffSaleDate
OffSaleDateA-platform
OffSaleDateDate
OffsetHDR
OffsetSDR
OffsetTime
OffsetTimeDigitized
OffsetTimeOriginal
OnSaleDate
OnSaleDateA-platform
OnSaleDateDate
OnSaleDay
OnSaleDayA-platform
OnSaleDayDay
OptionEndDate
Opto-ElectricConvFactor
OrganisationInImageCode
OrganisationInImageName
Organism
OrganismAssociatedOccurrences
OrganismAssociatedOrganisms
OrganismID
OrganismName
OrganismPreviousIdentifications
OrganismRemarks
OrganismScope
Organization
Orientation
OriginPlatform
OriginalCreateDateTime
OriginalDocumentID
OriginalFileName
OriginalImageHash
OriginalImageHashType
OriginalImageMD5
OriginatingProgram
OtherConditions
OtherConstraints
OtherImage
OtherImageInfo
OtherImageLength
OtherImageStart
OtherLicenseDocuments
OtherLicenseInfo
OtherLicenseRequirements
OutCue
OutCueScale
OutCueValue
OverrideLookVignette
Owner
OwnerName
PDFVersion
PLUSVersion
PMVersion
PageCount
PageImage
PageImageFormat
PageImageHeight
PageImagePageNumber
PageImageWidth
PageInfo
PageName
PageNumber
PageProgressionDirection
PageRange
PaintBasedCorrectionMasks
PaintBasedCorrections
PaintCorrectionActive
PaintCorrectionAmount
PaintCorrectionBlacks2012
PaintCorrectionBrightness
PaintCorrectionClarity
PaintCorrectionClarity2012
PaintCorrectionContrast
PaintCorrectionContrast2012
PaintCorrectionCorrectionName
PaintCorrectionCorrectionSyncID
PaintCorrectionDefringe
PaintCorrectionDehaze
PaintCorrectionExposure
PaintCorrectionExposure2012
PaintCorrectionHighlights2012
PaintCorrectionHue
PaintCorrectionLuminanceNoise
PaintCorrectionMaskAlpha
PaintCorrectionMaskAngle
PaintCorrectionMaskBottom
PaintCorrectionMaskCenterValue
PaintCorrectionMaskCenterWeight
PaintCorrectionMaskDabs
PaintCorrectionMaskFeather
PaintCorrectionMaskFlipped
PaintCorrectionMaskFlow
PaintCorrectionMaskFullX
PaintCorrectionMaskFullY
PaintCorrectionMaskInputDigest
PaintCorrectionMaskLeft
PaintCorrectionMaskMaskActive
PaintCorrectionMaskMaskBlendMode
PaintCorrectionMaskMaskDigest
PaintCorrectionMaskMaskInverted
PaintCorrectionMaskMaskName
PaintCorrectionMaskMaskSubType
PaintCorrectionMaskMaskSyncID
PaintCorrectionMaskMaskVersion
PaintCorrectionMaskMasks
PaintCorrectionMaskMasksAlpha
PaintCorrectionMaskMasksAngle
PaintCorrectionMaskMasksBottom
PaintCorrectionMaskMasksCenterValue
PaintCorrectionMaskMasksCenterWeight
PaintCorrectionMaskMasksDabs
PaintCorrectionMaskMasksFeather
PaintCorrectionMaskMasksFlipped
PaintCorrectionMaskMasksFlow
PaintCorrectionMaskMasksFullX
PaintCorrectionMaskMasksFullY
PaintCorrectionMaskMasksInputDigest
PaintCorrectionMaskMasksLeft
PaintCorrectionMaskMasksMaskActive
PaintCorrectionMaskMasksMaskBlendMode
PaintCorrectionMaskMasksMaskDigest
PaintCorrectionMaskMasksMaskInverted
PaintCorrectionMaskMasksMaskName
PaintCorrectionMaskMasksMaskSubType
PaintCorrectionMaskMasksMaskSyncID
PaintCorrectionMaskMasksMaskVersion
PaintCorrectionMaskMasksMidpoint
PaintCorrectionMaskMasksOrigin
PaintCorrectionMaskMasksPerimeterValue
PaintCorrectionMaskMasksRadius
PaintCorrectionMaskMasksReferencePoint
PaintCorrectionMaskMasksRight
PaintCorrectionMaskMasksRoundness
PaintCorrectionMaskMasksSizeX
PaintCorrectionMaskMasksSizeY
PaintCorrectionMaskMasksTop
PaintCorrectionMaskMasksValue
PaintCorrectionMaskMasksVersion
PaintCorrectionMaskMasksWhat
PaintCorrectionMaskMasksWholeImageArea
PaintCorrectionMaskMasksX
PaintCorrectionMaskMasksY
PaintCorrectionMaskMasksZeroX
PaintCorrectionMaskMasksZeroY
PaintCorrectionMaskMidpoint
PaintCorrectionMaskOrigin
PaintCorrectionMaskPerimeterValue
PaintCorrectionMaskRadius
PaintCorrectionMaskRange
PaintCorrectionMaskRangeAreaModels
PaintCorrectionMaskRangeAreaModelsColorSampleInfo
PaintCorrectionMaskRangeAreaModelsComponents
PaintCorrectionMaskRangeColorAmount
PaintCorrectionMaskRangeDepthFeather
PaintCorrectionMaskRangeDepthMax
PaintCorrectionMaskRangeDepthMin
PaintCorrectionMaskRangeInvert
PaintCorrectionMaskRangeLumFeather
PaintCorrectionMaskRangeLumMax
PaintCorrectionMaskRangeLumMin
PaintCorrectionMaskRangeLumRange
PaintCorrectionMaskRangeLuminanceDepthSampleInfo
PaintCorrectionMaskRangeSampleType
PaintCorrectionMaskRangeType
PaintCorrectionMaskRangeVersion
PaintCorrectionMaskReferencePoint
PaintCorrectionMaskRight
PaintCorrectionMaskRoundness
PaintCorrectionMaskSizeX
PaintCorrectionMaskSizeY
PaintCorrectionMaskTop
PaintCorrectionMaskValue
PaintCorrectionMaskVersion
PaintCorrectionMaskWhat
PaintCorrectionMaskWholeImageArea
PaintCorrectionMaskX
PaintCorrectionMaskY
PaintCorrectionMaskZeroX
PaintCorrectionMaskZeroY
PaintCorrectionMoire
PaintCorrectionRangeMask
PaintCorrectionRangeMaskAreaModels
PaintCorrectionRangeMaskAreaModelsColorSampleInfo
PaintCorrectionRangeMaskAreaModelsComponents
PaintCorrectionRangeMaskColorAmount
PaintCorrectionRangeMaskDepthFeather
PaintCorrectionRangeMaskDepthMax
PaintCorrectionRangeMaskDepthMin
PaintCorrectionRangeMaskInvert
PaintCorrectionRangeMaskLumFeather
PaintCorrectionRangeMaskLumMax
PaintCorrectionRangeMaskLumMin
PaintCorrectionRangeMaskLumRange
PaintCorrectionRangeMaskLuminanceDepthSampleInfo
PaintCorrectionRangeMaskSampleType
PaintCorrectionRangeMaskType
PaintCorrectionRangeMaskVersion
PaintCorrectionSaturation
PaintCorrectionShadows2012
PaintCorrectionSharpness
PaintCorrectionTemperature
PaintCorrectionTexture
PaintCorrectionTint
PaintCorrectionToningHue
PaintCorrectionToningSaturation
PaintCorrectionWhat
PaintCorrectionWhites2012
PanasonicRawVersion
PanasonicTitle
PanasonicTitle2
PanoramicStitchCameraMotion
PanoramicStitchMapType
PanoramicStitchPhi0
PanoramicStitchPhi1
PanoramicStitchTheta0
PanoramicStitchTheta1
Pantry
PantryInstanceID
ParametricDarks
ParametricHighlightSplit
ParametricHighlights
ParametricLights
ParametricMidtoneSplit
ParametricShadowSplit
ParametricShadows
ParentID
ParentMEID
ParentMediaEventID
ParentReference1
ParentReference2
ParentReference3
ParentReference4
ParentReference5
ParentReference6
PartOfCompilation
PatientBirthDate
PatientID
PatientName
PatientSex
People
Permissions
Permits
Person
PersonHeard
PersonHeardIdentifier
PersonHeardName
PersonInImage
PersonInImageCharacteristic
PersonInImageCvTermCvId
PersonInImageCvTermId
PersonInImageCvTermName
PersonInImageCvTermRefinedAbout
PersonInImageDescription
PersonInImageId
PersonInImageName
PersonInImageWDetails
Personality
PerspectiveAspect
PerspectiveHorizontal
PerspectiveRotate
PerspectiveScale
PerspectiveUpright
PerspectiveVertical
PerspectiveX
PerspectiveY
Photographer
PhotographicSensitivity
PicasawebGPhotoId
PickLabel
PipelineVersion
PitchAngle
PixelScale
Plane
PlaneBoundary
PlaneBoundaryVertexCount
PlaneExtentX
PlaneExtentZ
PlanePose
PlanePosePositionX
PlanePosePositionY
PlanePosePositionZ
PlanePoseRotationW
PlanePoseRotationX
PlanePoseRotationY
PlanePoseRotationZ
PlanePoseTimestamp
Planes
PlanningRef
PlanningRefIdentifier
PlanningRefName
PlanningRefRole
PlateNames
Platform
PortraitNote
PortraitRequest
PortraitVersion
Pose
PoseHeadingDegrees
PosePitchDegrees
PosePositionX
PosePositionY
PosePositionZ
PoseRollDegrees
PoseRotationW
PoseRotationX
PoseRotationY
PoseRotationZ
PoseTimestamp
PositionDescriptor
PostCropVignetteAmount
PostCropVignetteFeather
PostCropVignetteHighlightContrast
PostCropVignetteMidpoint
PostCropVignetteRoundness
PostCropVignetteStyle
Prefs
PreservedFileName
PreservedSpecimen
PreservedSpecimenMaterialSampleID
PresetType
Pressure
PreviewImage
PreviewImageLength
PreviewImageStart
PrimaryChromaticities
PrimaryFTP
PrintIM
PrivateRTKInfo
ProcessVersion
ProcessingSoftware
Producer
ProductCode
ProductID
ProductIDType
ProductInImage
ProductInImageDescription
ProductInImageGTIN
ProductInImageName
ProductInImageProductId
ProductOrServiceConstraints
Profession
Profile
ProfileCameraIndices
ProfileDynamicRange
ProfileType
Profiles
Prohibits
ProjectName
ProjectRef
ProjectRefPath
ProjectRefType
ProjectionType
PropertyReleaseID
PropertyReleaseStatus
PublicationDate
PublicationDateA-platform
PublicationDateDate
PublicationDisplayDate
PublicationDisplayDateA-platform
PublicationDisplayDateDate
PublicationEvent
PublicationEventDate
PublicationEventIdentifier
PublicationEventName
PublicationName
Publisher
PublishingFrequency
PullDown
RAWDevelopingSoftware
RPP
RangeMask
RangeMaskMapInfo
RangeMaskMapInfoLabMax
RangeMaskMapInfoLabMin
RangeMaskMapInfoLumEq
RangeMaskMapInfoRGBMax
RangeMaskMapInfoRGBMin
Rating
RatingPercent
RatingRegion
RatingRegionCity
RatingRegionCountryCode
RatingRegionCountryName
RatingRegionGPSAltitude
RatingRegionGPSLatitude
RatingRegionGPSLongitude
RatingRegionIdentifier
RatingRegionLocationId
RatingRegionLocationName
RatingRegionProvinceState
RatingRegionSublocation
RatingRegionWorldRegion
RatingScaleMaxValue
RatingScaleMinValue
RatingSourceLink
RatingValue
RatingValueLogoLink
RawFile
RawFileName
RawFormat
Rawrppused
RecDevice
RecDeviceAttLensDescription
RecDeviceManufacturer
RecDeviceModelName
RecDeviceOwnersDeviceId
RecDeviceSerialNumber
RecipeEndingPage
RecipePageRange
RecipeSource
RecipeStartingPage
RecipeTitle
RecommendedExposureIndex
Record
RecordBasisOfRecord
RecordCollectionCode
RecordCollectionID
RecordDataGeneralizations
RecordDatasetID
RecordDatasetName
RecordDynamicProperties
RecordInformationWithheld
RecordInstitutionCode
RecordInstitutionID
RecordOwnerInstitutionCode
RedBalance
RedEyeInfo
RedHue
RedSaturation
ReelName
Reference1
Reference2
Reference3
Reference4
Reference5
Reference6
ReferenceBlackWhite
RegionAppliedToDimensions
RegionAppliedToDimensionsH
RegionAppliedToDimensionsUnit
RegionAppliedToDimensionsW
RegionArea
RegionAreaD
RegionAreaH
RegionAreaUnit
RegionAreaW
RegionAreaX
RegionAreaY
RegionBarCodeValue
RegionConstraints
RegionDescription
RegionExtensions
RegionFocusUsage
RegionInfo
RegionInfoDateRegionsValid
RegionInfoMP
RegionInfoRegions
RegionList
RegionName
RegionPersonDisplayName
RegionPersonEmailDigest
RegionPersonLiveIdCID
RegionPersonSourceID
RegionRectangle
RegionRotation
RegionSeeAlso
RegionType
RegistryEntryRole
RegistryID
RegistryItemID
RegistryOrganisationID
RelatedResourceID
RelatedSoundFile
Relation
RelationshipAccordingTo
RelationshipEstablishedDate
RelationshipOfResource
RelationshipOfResourceID
RelationshipRemarks
RelativeAltitude
RelativePeakAudioFilePath
RelativeTimestamp
RelativeTimestampScale
RelativeTimestampValue
ReleaseDate
ReleaseReady
ReleaseTime
RenditionClass
RenditionOf
RenditionOfAlternatePaths
RenditionOfDocumentID
RenditionOfFilePath
RenditionOfFromPart
RenditionOfInstanceID
RenditionOfLastModifyDate
RenditionOfLastURL
RenditionOfLinkCategory
RenditionOfLinkForm
RenditionOfManageTo
RenditionOfManageUI
RenditionOfManager
RenditionOfManagerVariant
RenditionOfMaskMarkers
RenditionOfOriginalDocumentID
RenditionOfPartMapping
RenditionOfPlacedResolutionUnit
RenditionOfPlacedXResolution
RenditionOfPlacedYResolution
RenditionOfRenditionClass
RenditionOfRenditionParams
RenditionOfToPart
RenditionOfVersionID
RenditionParams
Requires
ResampleParams
ResampleParamsQuality
ResourceID
ResourceRelationship
ResourceRelationshipID
Restrictions
RetouchAreaFeather
RetouchAreaMaskAlpha
RetouchAreaMaskAngle
RetouchAreaMaskBottom
RetouchAreaMaskCenterValue
RetouchAreaMaskCenterWeight
RetouchAreaMaskDabs
RetouchAreaMaskFeather
RetouchAreaMaskFlipped
RetouchAreaMaskFlow
RetouchAreaMaskFullX
RetouchAreaMaskFullY
RetouchAreaMaskInputDigest
RetouchAreaMaskLeft
RetouchAreaMaskMaskActive
RetouchAreaMaskMaskBlendMode
RetouchAreaMaskMaskDigest
RetouchAreaMaskMaskInverted
RetouchAreaMaskMaskName
RetouchAreaMaskMaskSubType
RetouchAreaMaskMaskSyncID
RetouchAreaMaskMaskVersion
RetouchAreaMaskMasks
RetouchAreaMaskMasksAlpha
RetouchAreaMaskMasksAngle
RetouchAreaMaskMasksBottom
RetouchAreaMaskMasksCenterValue
RetouchAreaMaskMasksCenterWeight
RetouchAreaMaskMasksDabs
RetouchAreaMaskMasksFeather
RetouchAreaMaskMasksFlipped
RetouchAreaMaskMasksFlow
RetouchAreaMaskMasksFullX
RetouchAreaMaskMasksFullY
RetouchAreaMaskMasksInputDigest
RetouchAreaMaskMasksLeft
RetouchAreaMaskMasksMaskActive
RetouchAreaMaskMasksMaskBlendMode
RetouchAreaMaskMasksMaskDigest
RetouchAreaMaskMasksMaskInverted
RetouchAreaMaskMasksMaskName
RetouchAreaMaskMasksMaskSubType
RetouchAreaMaskMasksMaskSyncID
RetouchAreaMaskMasksMaskVersion
RetouchAreaMaskMasksMidpoint
RetouchAreaMaskMasksOrigin
RetouchAreaMaskMasksPerimeterValue
RetouchAreaMaskMasksRadius
RetouchAreaMaskMasksReferencePoint
RetouchAreaMaskMasksRight
RetouchAreaMaskMasksRoundness
RetouchAreaMaskMasksSizeX
RetouchAreaMaskMasksSizeY
RetouchAreaMaskMasksTop
RetouchAreaMaskMasksValue
RetouchAreaMaskMasksVersion
RetouchAreaMaskMasksWhat
RetouchAreaMaskMasksWholeImageArea
RetouchAreaMaskMasksX
RetouchAreaMaskMasksY
RetouchAreaMaskMasksZeroX
RetouchAreaMaskMasksZeroY
RetouchAreaMaskMidpoint
RetouchAreaMaskOrigin
RetouchAreaMaskPerimeterValue
RetouchAreaMaskRadius
RetouchAreaMaskRange
RetouchAreaMaskRangeAreaModels
RetouchAreaMaskRangeAreaModelsColorSampleInfo
RetouchAreaMaskRangeAreaModelsComponents
RetouchAreaMaskRangeColorAmount
RetouchAreaMaskRangeDepthFeather
RetouchAreaMaskRangeDepthMax
RetouchAreaMaskRangeDepthMin
RetouchAreaMaskRangeInvert
RetouchAreaMaskRangeLumFeather
RetouchAreaMaskRangeLumMax
RetouchAreaMaskRangeLumMin
RetouchAreaMaskRangeLumRange
RetouchAreaMaskRangeLuminanceDepthSampleInfo
RetouchAreaMaskRangeSampleType
RetouchAreaMaskRangeType
RetouchAreaMaskRangeVersion
RetouchAreaMaskReferencePoint
RetouchAreaMaskRight
RetouchAreaMaskRoundness
RetouchAreaMaskSizeX
RetouchAreaMaskSizeY
RetouchAreaMaskTop
RetouchAreaMaskValue
RetouchAreaMaskVersion
RetouchAreaMaskWhat
RetouchAreaMaskWholeImageArea
RetouchAreaMaskX
RetouchAreaMaskY
RetouchAreaMaskZeroX
RetouchAreaMaskZeroY
RetouchAreaMasks
RetouchAreaMethod
RetouchAreaOffsetY
RetouchAreaOpacity
RetouchAreaSeed
RetouchAreaSourceState
RetouchAreaSourceX
RetouchAreaSpotType
RetouchAreas
RetouchInfo
Reuse
ReuseAllowed
ReuseProhibited
Revision
RichTextComment
RightAlbedo
Rights
RightsAgent
RightsOwner
RollAngle
RoutedTo
RoutingDestinations
RoutingExclusions
RoutingNotes
RtkFlag
RtkStdHgt
RtkStdLat
RtkStdLon
SDRBlend
SDRBrightness
SDRContrast
SDRHighlights
SDRShadows
SDRWhites
SEMInfo
SamplePageRange
Saturation
SaturationAdjustmentAqua
SaturationAdjustmentBlue
SaturationAdjustmentGreen
SaturationAdjustmentMagenta
SaturationAdjustmentOrange
SaturationAdjustmentPurple
SaturationAdjustmentRed
SaturationAdjustmentYellow
SaveID
ScaleType
ScalingFactorHeight
Scene
SceneCaptureType
SceneReferred
SceneType
Season
SeasonIdentifier
SeasonName
SeasonNumber
SecondaryFTP
Section
SecurityClassification
SelfData
SelfTimerMode
SellingAgency
SensingMethod
SensitivityType
SequenceName
SequenceNumber
SequenceTotalNumber
SerialNumber
Series
SeriesDateTime
SeriesDescription
SeriesIdentifier
SeriesModality
SeriesName
SeriesNumber
SeriesTitle
ServingSize
Setting
ShadowTint
Shadows
Shadows2012
SharpenDetail
SharpenEdgeMasking
SharpenRadius
Sharpness
ShootID
ShortDescription
ShortName
ShotDate
ShotDay
ShotLocation
ShotLogData
ShotName
ShotNumber
ShotSize
ShownEvent
ShownEventIdentifier
ShownEventName
ShutterSpeedValue
SidecarForExtension
SkillLevel
SlideshowName
SlideshowNumber
SlideshowTotalNumber
Smoothness
Snapshot
SnapshotFormat
SnapshotHeightPixels
SnapshotImageRole
SnapshotLink
SnapshotLinkQualifier
SnapshotUsedVideoFrame
SnapshotUsedVideoFrameTimeFormat
SnapshotUsedVideoFrameTimeValue
SnapshotUsedVideoFrameValue
SnapshotWidthPixels
Snapshots
Software
SortName
Source
SourceCount
SourcePhotosCount
SpatialFrequencyResponse
SpatialFrequencyResponseColumns
SpatialFrequencyResponseNames
SpatialFrequencyResponseRows
SpatialFrequencyResponseValues
SpeakerPlacement
SpecialOccasion
SpecialTypeID
SpectralSensitivity
Spherical
SplitToningBalance
SplitToningHighlightHue
SplitToningHighlightSaturation
SplitToningShadowHue
SplitToningShadowSaturation
Sport
StandardOutputSensitivity
StartTimeSampleSize
StartTimeScale
StartTimecode
StartTimecodeTimeFormat
StartTimecodeTimeValue
StartTimecodeValue
StartingPage
State
Status
StereoMode
Stitched
StitchingSoftware
StorylineIdentifier
StreamReady
StreamType
StretchMode
StudyDateTime
StudyDescription
StudyID
StudyPhysician
StylePeriod
SubLabels1
SubLabels2
SubLabels3
SubLabels4
SubLabels5
SubSecTime
SubSecTimeDigitized
SubSecTimeOriginal
SubVersionFileName
SubVersionReference
SubVersions
Subject
SubjectArea
SubjectCode
SubjectDistance
SubjectDistanceRange
SubjectLocation
Subsection1
Subsection2
Subsection3
Subsection4
Subtitle
SupplementDisplayID
SupplementStartingPage
SupplementTitle
SupplementalCategories
SupplyChainSource
SupplyChainSourceIdentifier
SupplyChainSourceName
SupportsAmount
SupportsColor
SupportsHighDynamicRange
SupportsMonochrome
SupportsNormalDynamicRange
SupportsOutputReferred
SupportsSceneReferred
SwatchColorantA
SwatchColorantB
SwatchColorantBlack
SwatchColorantBlue
SwatchColorantCyan
SwatchColorantGray
SwatchColorantGreen
SwatchColorantL
SwatchColorantMagenta
SwatchColorantMode
SwatchColorantRed
SwatchColorantSwatchName
SwatchColorantTint
SwatchColorantType
SwatchColorantYellow
SwatchGroupName
SwatchGroupType
SwatchGroups
SwatchGroupsColorants
TIFFHandling
TStop
TagStructure
Tagged
TagsList
TakeNumber
TapeName
TargetAspectRatio
TargetPrinter
Taxon
TaxonAcceptedNameUsage
TaxonAcceptedNameUsageID
TaxonClass
TaxonConceptID
TaxonCultivarEpithet
TaxonFamily
TaxonGenus
TaxonHigherClassification
TaxonID
TaxonInfraspecificEpithet
TaxonKingdom
TaxonNameAccordingTo
TaxonNameAccordingToID
TaxonNamePublishedIn
TaxonNamePublishedInID
TaxonNamePublishedInYear
TaxonNomenclaturalCode
TaxonNomenclaturalStatus
TaxonOrder
TaxonOriginalNameUsage
TaxonOriginalNameUsageID
TaxonParentNameUsage
TaxonParentNameUsageID
TaxonPhylum
TaxonRank
TaxonRemarks
TaxonScientificName
TaxonScientificNameAuthorship
TaxonScientificNameID
TaxonSpecificEpithet
TaxonSubgenus
TaxonTaxonomicStatus
TaxonVerbatimTaxonRank
TaxonVernacularName
Teaser
Tempo
TemporalCoverage
TemporalCoverageFrom
TemporalCoverageTo
TermsAndConditionsText
TermsAndConditionsURL
TextLayerName
TextLayerText
TextLayers
Texture
ThumbnailFormat
ThumbnailHeight
ThumbnailImage
ThumbnailLength
ThumbnailOffset
ThumbnailWidth
Thumbnails
Ticker
TimeCodes
TimeCreated
TimePeriod
TimeScaleParams
TimeScaleParamsFrameOverlappingPercentage
TimeScaleParamsFrameSize
TimeScaleParamsQuality
TimeShot
TimeSignature
TimeStamp
TimeZoneOffset
Tint
Title
ToggleStyleAmount
ToggleStyleDigest
ToneCurve
ToneCurveBlue
ToneCurveGreen
ToneCurveName
ToneCurveName2012
ToneCurvePV2012
ToneCurvePV2012Blue
ToneCurvePV2012Green
ToneCurvePV2012Red
ToneCurveRed
ToneMapStrength
TrackNumber
Tracks
TracksFrameRate
TracksMarkers
TracksMarkersComment
TracksMarkersCuePointParams
TracksMarkersCuePointParamsKey
TracksMarkersCuePointParamsValue
TracksMarkersCuePointType
TracksMarkersDuration
TracksMarkersLocation
TracksMarkersName
TracksMarkersProbability
TracksMarkersSpeaker
TracksMarkersStartTime
TracksMarkersTarget
TracksMarkersType
TracksTrackName
TracksTrackType
Transcript
TranscriptLink
TranscriptLinkLink
TranscriptLinkLinkQualifier
Transformation
TransmissionReference
Trapped
Type
TypeStatus
URL
URLA-platform
URLUrl
UUID
UniqueCameraModel
Units
UprightCenterMode
UprightCenterNormX
UprightCenterNormY
UprightDependentDigest
UprightFocalLength35mm
UprightFocalMode
UprightFourSegmentsCount
UprightFourSegments_0
UprightFourSegments_1
UprightFourSegments_2
UprightFourSegments_3
UprightGuidedDependentDigest
UprightPreview
UprightTransformCount
UprightTransform_0
UprightTransform_1
UprightTransform_2
UprightTransform_3
UprightTransform_4
UprightTransform_5
UprightVersion
Urgency
Usage
UsageTerms
UseGuidelines
UsePanoramaViewer
UserComment
UserFields
UspsNumber
ValidCropCorners
VendorInfo
VendorInfoManufacturer
VendorInfoModel
VendorInfoNotes
VerbatimIdentification
Version
VersionID
VersionIdentifier
Versions
VersionsComments
VersionsEvent
VersionsEventAction
VersionsEventChanged
VersionsEventInstanceID
VersionsEventParameters
VersionsEventSoftwareAgent
VersionsEventWhen
VersionsModifier
VersionsModifyDate
VersionsVersion
Vibrance
VideoAlphaMode
VideoAlphaPremultipleColor
VideoAlphaPremultipleColorA
VideoAlphaPremultipleColorB
VideoAlphaPremultipleColorBlack
VideoAlphaPremultipleColorBlue
VideoAlphaPremultipleColorCyan
VideoAlphaPremultipleColorGray
VideoAlphaPremultipleColorGreen
VideoAlphaPremultipleColorL
VideoAlphaPremultipleColorMagenta
VideoAlphaPremultipleColorMode
VideoAlphaPremultipleColorRed
VideoAlphaPremultipleColorSwatchName
VideoAlphaPremultipleColorTint
VideoAlphaPremultipleColorType
VideoAlphaPremultipleColorYellow
VideoAlphaUnityIsTransparent
VideoBitrate
VideoBitrateMode
VideoColorSpace
VideoCompressor
VideoDisplayAspectRatio
VideoEncodingProfile
VideoFieldOrder
VideoFrameRate
VideoFrameSize
VideoFrameSizeH
VideoFrameSizeUnit
VideoFrameSizeW
VideoModDate
VideoPixelAspectRatio
VideoPixelDepth
VideoShotType
VideoShotTypeIdentifier
VideoShotTypeName
VideoStreamsCount
Viewpoint
VignetteAmount
VignetteCorrectionAlreadyApplied
VignetteMidpoint
VirtualFocalLength
VirtualImageXCenter
VirtualImageYCenter
VisualColor
VisualTechnique
Volume
WBBlueLevel
WBGreenLevel
WBRedLevel
WaterDepth
WebStatement
WeightedFlatSubject
What
WhiteBalance
WhiteBalance0
WhiteBalance1
WhiteBalance2
WhitePoint
Whites2012
WindowsAtom
WindowsAtomExtension
WindowsAtomInvocationFlags
WindowsAtomUncProjectPath
WordCount
WorkToDo
WorkflowTag
WorkflowTagCvId
WorkflowTagCvTermId
WorkflowTagCvTermName
WorkflowTagCvTermRefinedAbout
XMP
XMPToolkit
XPAuthor
XPComment
XPKeywords
XPSubject
XPTitle
XPosition
YPosition
Yield
