import React, {useState} from 'react';
import {Breadcrumbs, Button, ButtonGroup, DesktopChrome, MobileChrome, PageHeader, Select, type SelectOption} from '@tryghost/admin-x-design-system';
import {type OfficialTheme, type ThemeVariant} from '../../../providers/settings-app-provider';
import {type Theme, isDefaultOrLegacyTheme} from '@tryghost/admin-x-framework/api/themes';

const hasVariants = (theme: OfficialTheme) => theme.variants && theme.variants.length > 0;

const getAllVariants = (theme: OfficialTheme) : ThemeVariant[] => {
    const variants = [{
        image: theme.image,
        category: theme.category,
        previewUrl: theme.previewUrl
    }];

    if (theme.variants && theme.variants.length > 0) {
        variants.push(...theme.variants);
    }

    return variants;
};

const generateVariantOptionValue = (variant: ThemeVariant) => variant.category.toLowerCase();

const ThemePreview: React.FC<{
    selectedTheme?: OfficialTheme;
    isInstalling?: boolean;
    installedTheme?: Theme;
    onBack: () => void;
    onClose: () => void;
    onInstall?: () => void | Promise<void>;
}> = ({
    selectedTheme,
    isInstalling,
    installedTheme,
    onBack,
    onInstall
}) => {
    const [previewMode, setPreviewMode] = useState('desktop');
    const [selectedVariant, setSelectedVariant] = useState<SelectOption | undefined>(undefined);

    if (!selectedTheme) {
        return null;
    }

    let previewUrl = selectedTheme.previewUrl;

    const variantOptions = getAllVariants(selectedTheme).map((variant) => {
        return {
            label: variant.category,
            value: generateVariantOptionValue(variant)
        };
    });

    if (hasVariants(selectedTheme)) {
        if (selectedVariant === undefined) {
            setSelectedVariant(variantOptions[0]);
        }

        previewUrl = getAllVariants(selectedTheme).find(variant => generateVariantOptionValue(variant) === selectedVariant?.value)?.previewUrl || previewUrl;
    }

    let installButtonLabel = `Install ${selectedTheme.name}`;

    if (isInstalling) {
        installButtonLabel = 'Installing...';
    } else if (isDefaultOrLegacyTheme(selectedTheme) && !installedTheme?.active) {
        installButtonLabel = `Activate ${selectedTheme.name}`;
    } else if (installedTheme) {
        installButtonLabel = `Update ${selectedTheme.name}`;
    }

    const handleInstall = () => {
        // The parent component handles all limit checks and confirmation modals
        onInstall?.();
    };

    const left =
        <div className='flex items-center gap-2'>
            <Breadcrumbs
                activeItemClassName='hidden md:!block md:!visible'
                containerClassName='whitespace-nowrap'
                itemClassName='hidden md:!block md:!visible'
                items={[
                    {label: 'Change theme', onClick: onBack},
                    {label: selectedTheme.name}
                ]}
                separatorClassName='hidden md:!block md:!visible'
                backIcon
                onBack={onBack}
            />
            {hasVariants(selectedTheme) ?
                <>
                    <span className='hidden md:!visible md:!block'>–</span>
                    <Select
                        border={false}
                        containerClassName='text-sm font-bold'
                        controlClasses={{menu: 'w-24'}}
                        fullWidth={false}
                        options={variantOptions}
                        selectedOption={selectedVariant}
                        clearBg
                        onSelect={(option) => {
                            setSelectedVariant(option || undefined);
                        }}
                    />
                </> : null
            }
        </div>;

    const right =
        <div className='flex justify-end gap-8'>
            <ButtonGroup
                buttons={[
                    {
                        icon: 'laptop',
                        iconColorClass: (previewMode === 'desktop' ? 'text-black dark:text-green' : 'text-grey-500 dark:text-grey-600'),
                        link: true,
                        size: 'sm',
                        onClick: () => {
                            setPreviewMode('desktop');
                        }
                    },
                    {
                        icon: 'mobile',
                        iconColorClass: (previewMode === 'mobile' ? 'text-black dark:text-green' : 'text-grey-500 dark:text-grey-600'),
                        link: true,
                        size: 'sm',
                        onClick: () => {
                            setPreviewMode('mobile');
                        }
                    }
                ]}
            />
            <Button
                color='green'
                disabled={isInstalling}
                label={isInstalling ? 'Installing...' : installButtonLabel}
                onClick={handleInstall}
            />
        </div>;

    return (
        <div className='absolute inset-0 z-[100]'>
            <PageHeader containerClassName='bg-grey-50 dark:bg-black z-[100]' left={left} right={right} sticky={false} />
            <div className='flex h-[calc(100%-92px)] grow flex-col items-center justify-center bg-grey-50 dark:bg-black'>
                {previewMode === 'desktop' ?
                    <DesktopChrome>
                        <iframe
                            className='size-full'
                            src={previewUrl}
                            title='Theme preview'
                        />
                    </DesktopChrome>
                    :
                    <MobileChrome>
                        <iframe
                            className='size-full'
                            src={previewUrl}
                            title='Theme preview'
                        />
                    </MobileChrome>
                }
            </div>
        </div>
    );
};

export default ThemePreview;
