# See bottom of file for license and copyright information

=begin TML

---+ package Foswiki::Infix::Node

Base class for node types generated by Infix::Parser (and its subclasses). This
defines the interface used to construct nodes from the parser.

=cut

package Foswiki::Infix::Node;

use strict;
use warnings;
use Assert;

BEGIN {
    if ( $Foswiki::cfg{UseLocale} ) {
        require locale;
        import locale();
    }
}

# 1 for debug
use constant MONITOR => 0;

# Leaf token types
use constant {
    NAME   => 1,
    NUMBER => 2,
    STRING => 3,
    HASH   => 4,
    META   => 5,
};

=begin TML

---++ ClassMethod newNode( $o, @p ) -> \$if

Construct a new parse node (contract with Infix::Parser)

=cut

sub newNode {
    my $class = shift;
    my $op    = shift;
    my $this  = bless( {}, $class );
    @{ $this->{params} } = @_;
    $this->{op} = $op;
    return $this;
}

=begin TML

---++ ClassMethod newLeaf( $val, $type ) -> \$if

Construct a new terminal node (contract with Infix::Parser)
   * =$val= - the (scalar) value of the leaf
   * =$type= - one of NAME, NUMBER, STRING

=cut

sub newLeaf {
    my ( $class, $val, $type ) = @_;
    return newNode( $class, $type, $val );
}

=begin TML

---++ ObjectMethod convertToLeaf( $type, $val )

Convert the node into a terminal node.
   * =$type= - one of NAME, NUMBER, STRING
   * =$val= - the (scalar) value of the leaf

Not used by the Foswiki::Infix::Parser.

=cut

sub convertToLeaf {
    my ( $this, $type, $val ) = @_;
    $this->{op}     = $type;
    $this->{params} = [$val];
}

=begin TML

---++ ObjectMethod isLeaf() -> $boolean

Return true if this node was created by =newLeaf()= (or converted by =convertToLeaf=)

=cut

sub isLeaf {
    my $this = shift;
    return !ref( $this->{op} );
}

=begin TML

---++ ObjectMethod stringify() -> $string

Generate a string representation of the subtree, for reporting. The representation
generated by this function should be parseable, but is not guaranteed to be.

=cut

sub stringify {
    my $this = shift;

    unless ( ref( $this->{op} ) ) {
        if ( $this->{op} == STRING ) {
            ASSERT( ref( $this->{params}[0] ) eq '' ) if DEBUG;
            return "'$this->{params}[0]'";
        }
        else {
            if ( ref( $this->{params}[0] ) eq 'HASH' ) {
                return join( ',',
                    map { $_ . "=>'" . $this->{params}[0]->{$_} . "'" }
                    sort keys( %{ $this->{params}[0] } ) );
            }
            if ( ref( $this->{params}[0] ) eq 'Foswiki::Meta' ) {
                return $this->{params}[0]->web . '.'
                  . $this->{params}[0]->topic;
            }
            return $this->{params}[0];
        }
    }
    if ( $this->{op}->{arity} ) {
        use Foswiki::Query::Node;
        return
          $this->{op}->{name} . '{'
          . join( ',', map { stringify($_) } @{ $this->{params} } ) . '}';
    }
    else {
        $this->{op}->{name};
    }
}

1;
__END__
Author: Crawford Currie http://c-dot.co.uk

Foswiki - The Free and Open Source Wiki, http://foswiki.org/

Copyright (C) 2008-2011 Foswiki Contributors. Foswiki Contributors
are listed in the AUTHORS file in the root of this distribution.
NOTE: Please extend that file, not this notice.

Additional copyrights apply to some or all of the code in this
file as follows:

Copyright (C) 2005-2007 TWiki Contributors. All Rights Reserved.
TWiki Contributors are listed in the AUTHORS file in the root
of this distribution.

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version. For
more details read LICENSE in the root of this distribution.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.

As per the GPL, removal of this notice is prohibited.
